<?php

declare(strict_types=1);

require_once __DIR__ . '/../includes/helpers.php';
require_once __DIR__ . '/../includes/auth.php';

pemohon_require_login();

$currentUser = pemohon_current_user();
if ($currentUser === null) {
    pemohon_redirect('/pemohon');
}

$db = pemohon_db();
$userId = (int) $currentUser['id'];

$applicantSql = <<<'SQL'
    SELECT a.id AS applicant_id,
           a.nric,
           a.highest_qualification,
           a.institution,
           u.name,
           u.email
    FROM applicants a
    INNER JOIN users u ON u.id = a.user_id
    WHERE u.id = ?
    LIMIT 1
SQL;
$applicantStmt = $db->prepare($applicantSql);
if (!$applicantStmt) {
    throw new RuntimeException('Gagal mendapatkan data pemohon: ' . $db->error);
}
$applicantStmt->bind_param('i', $userId);
$applicantStmt->execute();
$applicantResult = $applicantStmt->get_result();
$applicant = $applicantResult->fetch_assoc() ?: null;
$applicantStmt->close();

$applications = [];
$applicationsSql = <<<'SQL'
    SELECT a.application_number,
           a.application_type,
           a.tauliah_type,
           a.status,
           a.current_stage,
           a.submitted_at,
           a.created_at,
           a.updated_at
    FROM applications a
    WHERE a.applicant_id = (
        SELECT id FROM applicants WHERE user_id = ? LIMIT 1
    )
    ORDER BY a.created_at DESC
SQL;
$applicationsStmt = $db->prepare($applicationsSql);
if ($applicationsStmt) {
    $applicationsStmt->bind_param('i', $userId);
    $applicationsStmt->execute();
    $applicationsResult = $applicationsStmt->get_result();
    while ($row = $applicationsResult->fetch_assoc()) {
        $applications[] = $row;
    }
    $applicationsStmt->close();
}

$typeLabels = [
    'new' => 'Permohonan Baharu',
    'renewal' => 'Pembaharuan Tempoh',
    'upgrade' => 'Naik Taraf Kategori',
    'replacement' => 'Penggantian Hilang/Rosak',
];

$tauliahLabels = [
    'agama' => 'Tauliah Agama',
    'alquran' => 'Tauliah Al-Quran',
    'both' => 'Tauliah Agama & Al-Quran',
];

$statusLabels = [
    'draft' => 'Draf',
    'submitted' => 'Dihantar',
    'under_review' => 'Dalam Semakan',
    'interview' => 'Temuduga',
    'approved' => 'Diluluskan',
    'rejected' => 'Ditolak',
    'kiv' => 'KIV',
    'cancelled' => 'Dibatalkan',
    'expired' => 'Tamat Tempoh',
];

$pageTitle = 'Permohonan Saya - eTauliah';
$flashMessage = $_SESSION['pemohon_message'] ?? '';
unset($_SESSION['pemohon_message']);

include __DIR__ . '/../header.php';
include __DIR__ . '/../sidebar.php';
?>
        <div class="page-wrapper">
            <div class="container-fluid">
                <div class="row page-titles">
                    <div class="col-md-6 col-12 align-self-center">
                        <h3 class="text-themecolor">Permohonan Saya</h3>
                        <ol class="breadcrumb">
                            <li class="breadcrumb-item"><a href="/pemohon/profile.php">Utama</a></li>
                            <li class="breadcrumb-item active">Permohonan Saya</li>
                        </ol>
                    </div>
                    <div class="col-md-6 col-12 align-self-center text-end">
                        <a href="/pemohon/applications/create.php" class="btn btn-info text-white">
                            <i class="ti-file"></i> Permohonan Baharu
                        </a>
                    </div>
                </div>

                <?php if ($applicant !== null && empty($applicant['nric'])): ?>
                    <div class="alert alert-warning">
                        <strong>Perhatian:</strong> Nombor Kad Pengenalan anda belum diisi. Sila lengkapkan maklumat profil
                        di <a href="/pemohon/update-profile.php" class="alert-link">Halaman Kemaskini Profil</a> sebelum menghantar permohonan baharu.
                    </div>
                <?php elseif ($applicant === null): ?>
                    <div class="alert alert-warning">
                        <strong>Perhatian:</strong> Maklumat pemohon belum lengkap. Sila kemaskini profil anda terlebih dahulu.
                    </div>
                <?php endif; ?>
                <?php if ($flashMessage !== ''): ?>
                    <div class="alert alert-success">
                        <?php echo htmlspecialchars($flashMessage, ENT_QUOTES, 'UTF-8'); ?>
                    </div>
                <?php endif; ?>

                <div class="row">
                    <div class="col-md-12">
                        <div class="card">
                            <div class="card-body">
                                <h4 class="card-title">Sejarah Permohonan</h4>
                                <div class="table-responsive">
                                    <table class="table table-striped datatable">
                                        <thead>
                                            <tr>
                                                <th>No. Permohonan</th>
                                                <th>Jenis Permohonan</th>
                                                <th>Jenis Tauliah</th>
                                                <th>Status</th>
                                                <th>Tarikh Hantar</th>
                                                <th>Kemaskini</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php if (!$applications): ?>
                                                <tr>
                                                    <td colspan="6" class="text-center">Tiada permohonan direkodkan.</td>
                                                </tr>
                                            <?php else: ?>
                                                <?php foreach ($applications as $application): ?>
                                                    <tr>
                                                        <td><?php echo htmlspecialchars($application['application_number'], ENT_QUOTES, 'UTF-8'); ?></td>
                                                        <td><?php echo htmlspecialchars($typeLabels[$application['application_type']] ?? $application['application_type'], ENT_QUOTES, 'UTF-8'); ?></td>
                                                        <td><?php echo htmlspecialchars($tauliahLabels[$application['tauliah_type']] ?? $application['tauliah_type'], ENT_QUOTES, 'UTF-8'); ?></td>
                                                        <td><?php echo htmlspecialchars($statusLabels[$application['status']] ?? strtoupper($application['status']), ENT_QUOTES, 'UTF-8'); ?></td>
                                                        <td>
                                                            <?php
                                                            $submittedAt = $application['submitted_at'] ?? null;
                                                            echo $submittedAt ? htmlspecialchars(date('d/m/Y', strtotime((string) $submittedAt)), ENT_QUOTES, 'UTF-8') : '-';
                                                            ?>
                                                        </td>
                                                        <td>
                                                            <?php
                                                            $updatedAt = $application['updated_at'] ?? null;
                                                            echo $updatedAt ? htmlspecialchars(date('d/m/Y H:i', strtotime((string) $updatedAt)), ENT_QUOTES, 'UTF-8') : '-';
                                                            ?>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
        </div><?php include __DIR__ . '/../footer.php'; ?>
