<?php

declare(strict_types=1);

require_once __DIR__ . '/includes/helpers.php';
require_once __DIR__ . '/includes/auth.php';
require_once __DIR__ . '/includes/applicant.php';

pemohon_require_login();

$currentUser = pemohon_current_user();
$userId = (int) ($currentUser['id'] ?? 0);
$profile = pemohon_fetch_profile($userId);
$metrics = pemohon_fetch_dashboard_metrics($userId);
$profileCompletion = pemohon_calculate_profile_completion($profile);
$microCredentialQrData = pemohon_dashboard_qr_payload($currentUser, $profile);

$statusChartJson = json_encode($metrics['status_chart'], JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP);
$trendChartJson = json_encode($metrics['trend_chart'], JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP);
$microCredentialQrJson = json_encode($microCredentialQrData, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP | JSON_UNESCAPED_UNICODE);

$pageTitle = 'Dashboard Pemohon - eTauliah';
include __DIR__ . '/header.php';
include __DIR__ . '/sidebar.php';
?>
<div class="page-wrapper">
    <div class="container-fluid">
        <style>
            .dashboard-charts .card {
                min-height: 360px;
            }
            #microCredentialQr {
                width: 180px;
                height: 180px;
                margin: 0 auto;
                border: 1px solid #eee;
                padding: 8px;
                border-radius: 8px;
                display: flex;
                align-items: center;
                justify-content: center;
                background-color: #fff;
            }
            #microCredentialQr canvas,
            #microCredentialQr img {
                width: 100% !important;
                height: 100% !important;
            }
        </style>
        <div class="row page-titles">
            <div class="col-md-5 align-self-center">
                <h4 class="text-themecolor mb-0">Dashboard Pemohon</h4>
                <small class="text-muted">Tinjauan ringkas status permohonan dan kemaskini profil anda.</small>
            </div>
        </div>

        <div class="row g-3">
            <div class="col-lg-3 col-md-6">
                <div class="card">
                    <div class="card-body">
                        <h6 class="text-muted">Jumlah Permohonan</h6>
                        <h3 class="mb-0"><?php echo number_format($metrics['total_applications']); ?></h3>
                        <span class="text-success small"><i class="ti-arrow-up"></i> Terkini dikemaskini <?php echo htmlspecialchars($metrics['last_update'] ?? 'tiada data', ENT_QUOTES, 'UTF-8'); ?></span>
                    </div>
                </div>
            </div>
            <div class="col-lg-3 col-md-6">
                <div class="card">
                    <div class="card-body">
                        <h6 class="text-muted">Dalam Proses</h6>
                        <h3 class="mb-0"><?php echo number_format($metrics['in_progress']); ?></h3>
                        <span class="text-info small">Sedang menunggu tindakan Pejabat Mufti Negeri Melaka</span>
                    </div>
                </div>
            </div>
            <div class="col-lg-3 col-md-6">
                <div class="card">
                    <div class="card-body">
                        <h6 class="text-muted">Diluluskan</h6>
                        <h3 class="mb-0"><?php echo number_format($metrics['approved']); ?></h3>
                        <span class="text-success small">Permohonan yang berjaya</span>
                    </div>
                </div>
            </div>
            <div class="col-lg-3 col-md-6">
                <div class="card">
                    <div class="card-body">
                        <h6 class="text-muted">Perlu Tindakan Anda</h6>
                        <h3 class="mb-0"><?php echo number_format($metrics['need_action']); ?></h3>
                        <span class="text-danger small">Lengkapkan maklumat atau dokumen</span>
                    </div>
                </div>
            </div>
        </div>

        <div class="row dashboard-charts g-3 mb-3">
            <div class="col-lg-6">
                <div class="card h-100">
                    <div class="card-body d-flex flex-column">
                        <div class="d-flex justify-content-between align-items-start mb-3">
                            <div>
                                <h4 class="card-title mb-1">Maklumat Mikro-Kredensial</h4>
                                <p class="text-muted mb-0 small">Gunakan QR ini untuk semakan sah Pejabat Mufti Negeri Melaka.</p>
                            </div>
                            <a href="/pemohon/profile.php" class="btn btn-sm btn-outline-primary">Kemaskini</a>
                        </div>
                        <div class="row align-items-center flex-grow-1 gy-3">
                            <div class="col-md-5 text-center">
                                <div id="microCredentialQr" aria-live="polite">
                                    <span class="text-muted small">Menjana kod QR...</span>
                                </div>
                                <p class="text-muted mt-2 mb-0 small">Imbas untuk akses rekod.</p>
                            </div>
                            <div class="col-md-7">
                                <div class="mb-2">
                                    <span class="text-muted small d-block">Nama Pemohon</span>
                                    <strong><?php echo htmlspecialchars($currentUser['name'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></strong>
                                </div>
                                <div class="mb-2">
                                    <span class="text-muted small d-block">No. Pengenalan</span>
                                    <strong><?php echo htmlspecialchars($profile['nric'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></strong>
                                </div>
                                <div class="mb-2">
                                    <span class="text-muted small d-block">Status Mikro-Kredensial</span>
                                    <span class="badge bg-success">Aktif</span>
                                </div>
                                <div>
                                    <span class="text-muted small d-block">Tempoh Sah</span>
                                    <strong><?php echo htmlspecialchars($metrics['micro_credential']['valid_until'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></strong>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-lg-6">
                <div class="card h-100">
                    <div class="card-body d-flex flex-column">
                        <h4 class="card-title mb-3">Notifikasi Terkini</h4>
                        <div class="list-group flex-grow-1">
                            <?php if (!$metrics['notifications']): ?>
                                <p class="text-muted text-center mb-0 align-self-center my-auto">Tiada notifikasi untuk anda.</p>
                            <?php else: ?>
                                <?php foreach ($metrics['notifications'] as $notification): ?>
                                    <a href="<?php echo htmlspecialchars($notification['link'], ENT_QUOTES, 'UTF-8'); ?>" class="list-group-item list-group-item-action d-flex gap-3 align-items-start">
                                        <div class="text-center">
                                            <span class="badge <?php echo htmlspecialchars($notification['badge'], ENT_QUOTES, 'UTF-8'); ?> mb-1"><?php echo htmlspecialchars($notification['type'], ENT_QUOTES, 'UTF-8'); ?></span>
                                            <div class="text-muted small"><?php echo htmlspecialchars($notification['date'], ENT_QUOTES, 'UTF-8'); ?></div>
                                        </div>
                                        <div>
                                            <h6 class="mb-1"><?php echo htmlspecialchars($notification['title'], ENT_QUOTES, 'UTF-8'); ?></h6>
                                            <p class="mb-0 text-muted small"><?php echo htmlspecialchars($notification['excerpt'], ENT_QUOTES, 'UTF-8'); ?></p>
                                        </div>
                                    </a>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                        <div class="mt-3 text-end">
                            <a href="/pemohon/notifications.php" class="btn btn-outline-secondary btn-sm">Lihat Semua Notifikasi</a>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <div class="col-lg-5">
                <div class="card">
                    <div class="card-body">
                        <h4 class="card-title">Lengkapkan Profil Anda</h4>
                        <p class="text-muted mb-3">Isikan maklumat peribadi untuk mempercepatkan proses semakan.</p>
                        <div class="progress mb-2" style="height: 20px;">
                            <div class="progress-bar bg-info" role="progressbar" style="width: <?php echo (int) $profileCompletion['percent']; ?>%;" aria-valuemin="0" aria-valuemax="100" aria-valuenow="<?php echo (int) $profileCompletion['percent']; ?>">
                                <?php echo (int) $profileCompletion['percent']; ?>%
                            </div>
                        </div>
                        <?php if (!empty($profileCompletion['missing'])): ?>
                            <p class="text-muted small mb-2">Maklumat yang masih belum dilengkapkan:</p>
                            <ul class="text-muted small ps-3 mb-0">
                                <?php foreach ($profileCompletion['missing'] as $missingField): ?>
                                    <li><?php echo htmlspecialchars($missingField, ENT_QUOTES, 'UTF-8'); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        <?php else: ?>
                            <p class="text-success mb-0">Tahniah! Profil anda lengkap.</p>
                        <?php endif; ?>
                        <div class="mt-3">
                            <a href="/pemohon/profile.php" class="btn btn-outline-primary btn-sm">Kemaskini Profil</a>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-lg-7">
                <div class="card">
                    <div class="card-body">
                        <h4 class="card-title">Aktiviti Permohonan Terkini</h4>
                        <div class="table-responsive">
                            <table class="table table-striped mb-0">
                                <thead>
                                    <tr>
                                        <th>No. Permohonan</th>
                                        <th>Status</th>
                                        <th>Kemaskini</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (!$metrics['recent_applications']): ?>
                                        <tr>
                                            <td colspan="3" class="text-center text-muted">Tiada aktiviti terkini.</td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($metrics['recent_applications'] as $recent): ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($recent['application_number'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
                                                <td><span class="badge <?php echo htmlspecialchars($recent['status_badge'], ENT_QUOTES, 'UTF-8'); ?>"><?php echo htmlspecialchars($recent['status_label'], ENT_QUOTES, 'UTF-8'); ?></span></td>
                                                <td><?php echo htmlspecialchars($recent['updated_at_display'], ENT_QUOTES, 'UTF-8'); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                        <div class="mt-3">
                            <a href="/pemohon/applications/index.php" class="btn btn-primary btn-sm">Lihat Semua Permohonan</a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<?php
ob_start();
?>
<script src="<?php echo htmlspecialchars(pemohon_asset('assets/js/qrcode.min.js'), ENT_QUOTES, 'UTF-8'); ?>"></script>
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function () {
    var statusData = <?php echo $statusChartJson; ?>;
    var trendData = <?php echo $trendChartJson; ?>;
    var qrPayload = <?php echo $microCredentialQrJson; ?>;
    var qrElement = document.getElementById('microCredentialQr');

    if (qrElement) {
        if (window.QRCode && qrPayload) {
            qrElement.innerHTML = '';
            new QRCode(qrElement, {
                text: qrPayload,
                width: 164,
                height: 164,
                correctLevel: QRCode.CorrectLevel.M
            });
        } else if (!window.QRCode) {
            qrElement.innerHTML = '<span class="text-danger small">Kod QR tidak dapat dimuatkan.</span>';
        } else {
            qrElement.innerHTML = '<span class="text-muted small">Maklumat tidak mencukupi.</span>';
        }
    }

    if (window.Chart && document.getElementById('statusChart') && statusData.data.length > 0) {
        new Chart(document.getElementById('statusChart'), {
            type: 'doughnut',
            data: {
                labels: statusData.labels,
                datasets: [{
                    data: statusData.data,
                    backgroundColor: ['#26c6da', '#ffb22b', '#5c6bc0', '#009688', '#ef5350', '#607d8b', '#26a69a'],
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });
    }

    if (window.Chart && document.getElementById('trendChart') && trendData.data.length > 0) {
        new Chart(document.getElementById('trendChart'), {
            type: 'line',
            data: {
                labels: trendData.labels,
                datasets: [{
                    label: 'Permohonan',
                    data: trendData.data,
                    borderColor: '#5e72e4',
                    backgroundColor: 'rgba(94,114,228,0.15)',
                    fill: true,
                    tension: 0.3,
                    pointRadius: 4
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: { display: false }
                },
                scales: {
                    y: { beginAtZero: true, precision: 0 }
                }
            }
        });
    }
});
</script>
<?php
$pageInlineScripts = ob_get_clean();
include __DIR__ . '/footer.php';

function pemohon_fetch_dashboard_metrics(int $userId): array
{
    $default = [
        'total_applications' => 0,
        'in_progress' => 0,
        'approved' => 0,
        'need_action' => 0,
        'status_chart' => ['labels' => [], 'data' => [], 'total' => 0],
        'trend_chart' => ['labels' => [], 'data' => [], 'total' => 0],
        'recent_applications' => [],
        'last_update' => 'Tiada',
    ];

    $applicantId = pemohon_get_applicant_id($userId);
    if (!$applicantId) {
        return $default;
    }

    $db = pemohon_db();
    $statusCounts = [];
    $statusStmt = $db->prepare('SELECT status, COUNT(*) AS total FROM applications WHERE applicant_id = ? GROUP BY status');
    if ($statusStmt) {
        $statusStmt->bind_param('i', $applicantId);
        $statusStmt->execute();
        $result = $statusStmt->get_result();
        while ($row = $result->fetch_assoc()) {
            $statusCounts[(string) $row['status']] = (int) $row['total'];
        }
        $statusStmt->close();
    }

    $totalApplications = array_sum($statusCounts);
    $inProgressStatuses = ['submitted', 'under_review', 'interview', 'kiv'];
    $needActionStatuses = ['draft', 'rejected', 'cancelled', 'expired'];

    $inProgress = array_sum(array_intersect_key($statusCounts, array_flip($inProgressStatuses)));
    $approved = $statusCounts['approved'] ?? 0;
    $needAction = array_sum(array_intersect_key($statusCounts, array_flip($needActionStatuses)));

    $statusLabels = pemohon_dashboard_status_labels();
    $statusChartLabels = [];
    $statusChartData = [];
    foreach ($statusLabels as $key => $label) {
        $count = $statusCounts[$key] ?? 0;
        if ($count > 0) {
            $statusChartLabels[] = $label;
            $statusChartData[] = $count;
        }
    }

    $trendLabels = [];
    $trendCounts = [];
    for ($i = 5; $i >= 0; $i--) {
        $monthKey = date('Y-m', strtotime("-{$i} months"));
        $trendLabels[] = date('M Y', strtotime($monthKey . '-01'));
        $trendCounts[$monthKey] = 0;
    }
    $trendStmt = $db->prepare('SELECT DATE_FORMAT(created_at, "%Y-%m") AS ym, COUNT(*) AS total FROM applications WHERE applicant_id = ? AND created_at >= DATE_SUB(DATE(NOW()), INTERVAL 6 MONTH) GROUP BY ym ORDER BY ym ASC');
    if ($trendStmt) {
        $trendStmt->bind_param('i', $applicantId);
        $trendStmt->execute();
        $result = $trendStmt->get_result();
        while ($row = $result->fetch_assoc()) {
            $ym = (string) $row['ym'];
            if (array_key_exists($ym, $trendCounts)) {
                $trendCounts[$ym] = (int) $row['total'];
            }
        }
        $trendStmt->close();
    }
    $trendData = array_values($trendCounts);

    $recent = [];
    $recentStmt = $db->prepare('SELECT application_number, status, updated_at FROM applications WHERE applicant_id = ? ORDER BY COALESCE(updated_at, created_at) DESC LIMIT 5');
    if ($recentStmt) {
        $recentStmt->bind_param('i', $applicantId);
        $recentStmt->execute();
        $result = $recentStmt->get_result();
        while ($row = $result->fetch_assoc()) {
            $statusKey = (string) ($row['status'] ?? 'pending');
            $recent[] = [
                'application_number' => $row['application_number'] ?? '-',
                'status_label' => pemohon_dashboard_status_labels()[$statusKey] ?? ucfirst(str_replace('_', ' ', $statusKey)),
                'status_badge' => pemohon_dashboard_status_badge($statusKey),
                'updated_at_display' => $row['updated_at'] ? date('d/m/Y H:i', strtotime((string) $row['updated_at'])) : '-',
            ];
        }
        $recentStmt->close();
    }

    $lastUpdate = '-';
    if ($recent) {
        $lastUpdate = $recent[0]['updated_at_display'];
    }

    return [
        'total_applications' => $totalApplications,
        'in_progress' => $inProgress,
        'approved' => $approved,
        'need_action' => $needAction,
        'status_chart' => [
            'labels' => $statusChartLabels,
            'data' => $statusChartData,
            'total' => array_sum($statusChartData),
        ],
        'trend_chart' => [
            'labels' => $trendLabels,
            'data' => $trendData,
            'total' => array_sum($trendData),
        ],
        'micro_credential' => [
            'valid_until' => $profile['micro_credential_valid_until'] ?? date('d/m/Y', strtotime('+1 year')),
        ],
        'notifications' => pemohon_dashboard_notifications($userId),
        'recent_applications' => $recent,
        'last_update' => $lastUpdate,
    ];
}

function pemohon_dashboard_status_labels(): array
{
    return [
        'draft' => 'Draf',
        'submitted' => 'Dihantar',
        'under_review' => 'Dalam Semakan',
        'interview' => 'Temuduga',
        'approved' => 'Diluluskan',
        'rejected' => 'Ditolak',
        'kiv' => 'KIV',
        'cancelled' => 'Dibatalkan',
        'expired' => 'Tamat Tempoh',
    ];
}

function pemohon_dashboard_status_badge(string $status): string
{
    return match ($status) {
        'approved' => 'badge bg-success',
        'submitted', 'under_review', 'interview', 'kiv' => 'badge bg-info',
        'draft' => 'badge bg-secondary',
        'rejected', 'cancelled', 'expired' => 'badge bg-danger',
        default => 'badge bg-primary',
    };
}

function pemohon_calculate_profile_completion(?array $profile): array
{
    $fields = [
        'nric' => 'No. Kad Pengenalan',
        'phone' => 'No. Telefon',
        'date_of_birth' => 'Tarikh Lahir',
        'occupation' => 'Pekerjaan',
        'address_line1' => 'Alamat',
        'city' => 'Bandar',
        'state' => 'Negeri',
        'postcode' => 'Poskod',
        'country' => 'Negara',
        'biography' => 'Biografi',
    ];

    if (!$profile) {
        return [
            'percent' => 0,
            'filled' => 0,
            'total' => count($fields),
            'missing' => array_values($fields),
        ];
    }

    $filled = 0;
    $missing = [];
    foreach ($fields as $key => $label) {
        $value = trim((string) ($profile[$key] ?? ''));
        if ($value === '') {
            $missing[] = $label;
        } else {
            $filled++;
        }
    }

    $total = count($fields);
    $percent = $total > 0 ? (int) round(($filled / $total) * 100) : 0;

    return [
        'percent' => $percent,
        'filled' => $filled,
        'total' => $total,
        'missing' => $missing,
    ];
}
function pemohon_dashboard_notifications(int $userId): array
{
    $db = pemohon_db();
    $items = [];
    $sql = <<<SQL
SELECT a.id,
       a.title,
       a.body,
       a.audience,
       COALESCE(a.published_from, a.created_at) AS published_at
FROM announcements AS a
LEFT JOIN announcement_targets AS t ON t.announcement_id = a.id
WHERE a.is_active = 1
  AND (a.published_from IS NULL OR a.published_from <= NOW())
  AND (a.published_until IS NULL OR a.published_until >= NOW())
  AND (
        a.audience = 'all'
        OR a.audience = 'applicant'
        OR (a.audience = 'custom' AND t.target_role = 'applicant')
      )
GROUP BY a.id
ORDER BY published_at DESC
LIMIT 5
SQL;
    $stmt = $db->prepare($sql);
    if ($stmt && $stmt->execute()) {
        $result = $stmt->get_result();
        while ($row = $result->fetch_assoc()) {
            $audience = $row['audience'] ?? 'applicant';
            $badge = 'badge bg-primary';
            $type = 'Pemohon';
            if ($audience === 'all') {
                $badge = 'badge bg-secondary';
                $type = 'Umum';
            } elseif ($audience === 'custom') {
                $badge = 'badge bg-info';
                $type = 'Sasaran';
            }

            $items[] = [
                'title' => $row['title'] ?? 'Notifikasi',
                'excerpt' => mb_strimwidth(strip_tags((string) ($row['body'] ?? '')), 0, 90, '...'),
                'date' => $row['published_at'] ? date('d/m/Y', strtotime((string) $row['published_at'])) : '-',
                'type' => $type,
                'badge' => $badge,
                'link' => '/pemohon/notifications.php',
            ];
        }
        $stmt->close();
    }

    return $items;
}

function pemohon_dashboard_qr_payload(?array $user, ?array $profile): string
{
    $name = trim((string) ($user['name'] ?? ''));
    if ($name === '') {
        $name = 'Pemohon eTauliah';
    }

    $nric = trim((string) ($profile['nric'] ?? ($user['nric'] ?? '')));
    if ($nric === '') {
        $nric = 'Tidak tersedia';
    }

    $phone = trim((string) ($profile['phone'] ?? ($user['phone'] ?? '')));
    if ($phone === '') {
        $phone = 'Tidak tersedia';
    }

    $email = trim((string) ($user['email'] ?? ''));
    if ($email === '') {
        $email = 'Tidak tersedia';
    }

    $lines = [
        'Nama: ' . $name,
        'MyKad: ' . $nric,
        'Telefon: ' . $phone,
        'Emel: ' . $email,
    ];

    return implode("\n", $lines);
}
