<?php

declare(strict_types=1);

require_once __DIR__ . '/includes/helpers.php';
require_once __DIR__ . '/includes/auth.php';
require_once __DIR__ . '/includes/documents.php';

pemohon_require_login();

$currentUser = pemohon_current_user();
if ($currentUser === null) {
    http_response_code(401);
    exit;
}

$userId = (int) ($currentUser['id'] ?? 0);
if ($userId <= 0) {
    http_response_code(401);
    exit;
}

$method = strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');

if ($method === 'GET') {
    header('Content-Type: application/json; charset=utf-8');

    try {
        $documents = pemohon_fetch_documents($userId);
    } catch (Throwable $exception) {
        http_response_code(500);
        echo json_encode([
            'message' => 'Gagal memuatkan senarai dokumen.',
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    $items = array_map(static function (array $row): array {
        return [
            'id' => (int) ($row['id'] ?? 0),
            'title' => (string) ($row['title'] ?? ''),
            'description' => $row['description'] ?? null,
            'original_filename' => (string) ($row['original_filename'] ?? ''),
            'mime_type' => (string) ($row['mime_type'] ?? ''),
            'file_size' => (int) ($row['file_size'] ?? 0),
            'created_at' => $row['created_at'] ?? null,
            'updated_at' => $row['updated_at'] ?? null,
            'view_url' => 'document-view.php?id=' . urlencode((string) ($row['id'] ?? 0)),
        ];
    }, $documents);

    echo json_encode([
        'items' => $items,
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

if ($method !== 'POST') {
    header('Content-Type: application/json; charset=utf-8');
    http_response_code(405);
    echo json_encode([
        'message' => 'Kaedah tidak dibenarkan.',
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

header('Content-Type: application/json; charset=utf-8');

$token = $_POST['_token'] ?? null;
if (!pemohon_validate_csrf_token(is_string($token) ? $token : null)) {
    http_response_code(419);
    echo json_encode([
        'message' => 'Sesi tidak sah. Sila muat semula halaman.',
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$title = trim((string) ($_POST['title'] ?? ''));
$description = trim((string) ($_POST['description'] ?? ''));

if ($title === '') {
    http_response_code(422);
    echo json_encode([
        'message' => 'Sila masukkan tajuk dokumen.',
        'errors' => ['title' => 'Tajuk dokumen diperlukan.'],
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

if (!isset($_FILES['document']['error']) || $_FILES['document']['error'] !== UPLOAD_ERR_OK) {
    http_response_code(422);
    echo json_encode([
        'message' => 'Sila pilih fail PDF untuk dimuat naik.',
        'errors' => ['document' => 'Fail tidak sah atau gagal dimuat naik.'],
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$file = $_FILES['document'];

if (!is_uploaded_file($file['tmp_name'])) {
    http_response_code(400);
    echo json_encode([
        'message' => 'Muat naik fail tidak sah.',
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$maxFileSize = 15 * 1024 * 1024; // 15MB
$fileSize = (int) ($file['size'] ?? 0);
if ($fileSize <= 0 || $fileSize > $maxFileSize) {
    http_response_code(422);
    echo json_encode([
        'message' => 'Saiz fail melebihi had maksimum 15MB.',
        'errors' => ['document' => 'Saiz fail melebihi had maksimum 15MB.'],
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$originalName = (string) ($file['name'] ?? 'dokumen.pdf');
$extension = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));
if ($extension !== 'pdf') {
    http_response_code(422);
    echo json_encode([
        'message' => 'Hanya fail PDF dibenarkan.',
        'errors' => ['document' => 'Hanya fail PDF dibenarkan.'],
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$finfo = new finfo(FILEINFO_MIME_TYPE);
$detectedMime = $finfo->file($file['tmp_name']) ?: '';
if (stripos($detectedMime, 'pdf') === false) {
    http_response_code(422);
    echo json_encode([
        'message' => 'Fail mestilah dalam format PDF.',
        'errors' => ['document' => 'Fail mestilah dalam format PDF.'],
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$applicantId = pemohon_get_applicant_id($userId);
if ($applicantId === null) {
    http_response_code(400);
    echo json_encode([
        'message' => 'Maklumat pemohon tidak ditemui.',
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$relativeDirectory = 'userfiles/pemohon/' . $userId . '/documents';
$normalizedDirectory = str_replace('\\', '/', $relativeDirectory);
$absoluteDirectory = dirname(__DIR__) . '/' . $normalizedDirectory;

if (!is_dir($absoluteDirectory) && !mkdir($absoluteDirectory, 0775, true) && !is_dir($absoluteDirectory)) {
    http_response_code(500);
    echo json_encode([
        'message' => 'Tidak dapat menyediakan direktori simpanan dokumen.',
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$storedFilename = 'dokumen-' . date('Ymd-His') . '-' . bin2hex(random_bytes(4)) . '.pdf';
$relativePath = $normalizedDirectory . '/' . $storedFilename;
$absolutePath = dirname(__DIR__) . '/' . $relativePath;

if (!move_uploaded_file($file['tmp_name'], $absolutePath)) {
    @unlink($absolutePath);
    http_response_code(500);
    echo json_encode([
        'message' => 'Gagal menyimpan dokumen.',
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$db = pemohon_db();

$sql = 'INSERT INTO applicant_documents
            (applicant_id, title, description, original_filename, stored_path, mime_type, file_size)
        VALUES (?, ?, ?, ?, ?, ?, ?)';

$stmt = $db->prepare($sql);
if (!$stmt) {
    @unlink($absolutePath);
    $errorMessage = 'Gagal menyediakan simpanan dokumen.';
    if ((int) $db->errno === 1146) {
        $errorMessage = 'Struktur pangkalan data belum dikemaskini. Sila jalankan skrip migrasi untuk jadual applicant_documents.';
    }
    http_response_code(500);
    echo json_encode([
        'message' => $errorMessage,
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$nullableDescription = $description !== '' ? $description : null;
$storedPath = str_replace('\\', '/', $relativePath);
$stmt->bind_param(
    'isssssi',
    $applicantId,
    $title,
    $nullableDescription,
    $originalName,
    $storedPath,
    $detectedMime,
    $fileSize
);

if (!$stmt->execute()) {
    $stmt->close();
    @unlink($absolutePath);
    http_response_code(500);
    echo json_encode([
        'message' => 'Gagal menyimpan rekod dokumen.',
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$documentId = (int) $stmt->insert_id;
$stmt->close();

echo json_encode([
    'message' => 'Dokumen berjaya dimuat naik.',
    'item' => [
        'id' => $documentId,
        'title' => $title,
        'description' => $nullableDescription,
        'original_filename' => $originalName,
        'mime_type' => $detectedMime,
        'file_size' => $fileSize,
        'view_url' => 'document-view.php?id=' . urlencode((string) $documentId),
    ],
], JSON_UNESCAPED_UNICODE);
exit;
