<?php

declare(strict_types=1);

require_once __DIR__ . '/db.php';
require_once __DIR__ . '/applicant.php';

/**
 * @return array<int, array<string, mixed>>
 */
function pemohon_fetch_educations(int $userId): array
{
    $db = pemohon_db();
    $sql = 'SELECT e.id, e.applicant_id, e.institution, e.qualification, e.field_of_study,
                   e.display_order, e.is_highest, e.start_year, e.end_year, e.description,
                   e.created_at, e.updated_at
            FROM applicant_educations e
            INNER JOIN applicants a ON a.id = e.applicant_id
            WHERE a.user_id = ?
            ORDER BY e.display_order ASC, e.id ASC';

    $stmt = $db->prepare($sql);
    if (!$stmt) {
        if ((int) $db->errno === 1146) {
            return [];
        }
        throw new \RuntimeException('Failed to prepare statement: ' . $db->error);
    }

    $stmt->bind_param('i', $userId);
    $stmt->execute();
    $result = $stmt->get_result();

    $items = [];
    while ($row = $result->fetch_assoc()) {
        $items[] = $row;
    }
    $stmt->close();

    return $items;
}

function pemohon_find_education(int $userId, int $educationId): ?array
{
    if ($educationId <= 0) {
        return null;
    }

    $db = pemohon_db();
    $sql = 'SELECT e.id, e.applicant_id, e.institution, e.qualification, e.field_of_study,
                   e.display_order, e.is_highest, e.start_year, e.end_year, e.description,
                   e.created_at, e.updated_at
            FROM applicant_educations e
            INNER JOIN applicants a ON a.id = e.applicant_id
            WHERE e.id = ? AND a.user_id = ?
            LIMIT 1';

    $stmt = $db->prepare($sql);
    if (!$stmt) {
        if ((int) $db->errno === 1146) {
            return null;
        }
        throw new \RuntimeException('Failed to prepare statement: ' . $db->error);
    }

    $stmt->bind_param('ii', $educationId, $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc() ?: null;
    $stmt->close();

    return $row;
}

/**
 * @param array<string, mixed> $data
 */
function pemohon_insert_education(int $userId, array $data): ?array
{
    $applicantId = pemohon_get_applicant_id($userId);
    if ($applicantId === null) {
        return null;
    }

    $db = pemohon_db();

    $orderResult = $db->prepare('SELECT COALESCE(MAX(display_order), 0) FROM applicant_educations WHERE applicant_id = ?');
    if (!$orderResult) {
        throw new \RuntimeException('Failed to prepare order lookup: ' . $db->error);
    }
    $orderResult->bind_param('i', $applicantId);
    $orderResult->execute();
    $orderResult->bind_result($currentMax);
    $orderResult->fetch();
    $orderResult->close();

    $nextOrder = ((int) $currentMax) + 1;

    $sql = 'INSERT INTO applicant_educations
                (applicant_id, institution, qualification, field_of_study, start_year, end_year, description, display_order, is_highest)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)';

    $stmt = $db->prepare($sql);
    if (!$stmt) {
        if ((int) $db->errno === 1146) {
            return null;
        }
        throw new \RuntimeException('Failed to prepare statement: ' . $db->error);
    }

    $institution = (string) ($data['institution'] ?? '');
    $qualification = (string) ($data['qualification'] ?? '');
    $field = isset($data['field_of_study']) && $data['field_of_study'] !== null
        ? (string) $data['field_of_study']
        : null;
    $startYear = isset($data['start_year']) && $data['start_year'] !== null
        ? (int) $data['start_year']
        : null;
    $endYear = isset($data['end_year']) && $data['end_year'] !== null
        ? (int) $data['end_year']
        : null;
    $description = isset($data['description']) && $data['description'] !== null
        ? (string) $data['description']
        : null;
    $isHighest = !empty($data['is_highest']) ? 1 : 0;

    $stmt->bind_param(
        'isssiisii',
        $applicantId,
        $institution,
        $qualification,
        $field,
        $startYear,
        $endYear,
        $description,
        $nextOrder,
        $isHighest
    );

    if (!$stmt->execute()) {
        $stmt->close();
        throw new \RuntimeException('Failed to insert education: ' . $stmt->error);
    }

    $insertId = (int) $stmt->insert_id;
    $stmt->close();

    if ($isHighest === 1) {
        pemohon_mark_highest_education($db, $applicantId, $insertId);
    } else {
        pemohon_ensure_highest_exists($db, $applicantId);
    }

    return pemohon_find_education($userId, $insertId);
}

function pemohon_update_education(int $userId, int $educationId, array $data): ?array
{
    if ($educationId <= 0) {
        return null;
    }

    $applicantId = pemohon_get_applicant_id($userId);
    if ($applicantId === null) {
        return null;
    }

    $db = pemohon_db();
    $sql = 'UPDATE applicant_educations
            SET institution = ?,
                qualification = ?,
                field_of_study = ?,
                start_year = ?,
                end_year = ?,
                description = ?,
                is_highest = ?
            WHERE id = ? AND applicant_id = ?';

    $stmt = $db->prepare($sql);
    if (!$stmt) {
        throw new \RuntimeException('Failed to prepare statement: ' . $db->error);
    }

    $institution = (string) ($data['institution'] ?? '');
    $qualification = (string) ($data['qualification'] ?? '');
    $field = isset($data['field_of_study']) && $data['field_of_study'] !== null
        ? (string) $data['field_of_study']
        : null;
    $startYear = isset($data['start_year']) && $data['start_year'] !== null
        ? (int) $data['start_year']
        : null;
    $endYear = isset($data['end_year']) && $data['end_year'] !== null
        ? (int) $data['end_year']
        : null;
    $description = isset($data['description']) && $data['description'] !== null
        ? (string) $data['description']
        : null;
    $isHighest = !empty($data['is_highest']) ? 1 : 0;

    $stmt->bind_param(
        'sssiisiii',
        $institution,
        $qualification,
        $field,
        $startYear,
        $endYear,
        $description,
        $isHighest,
        $educationId,
        $applicantId
    );

    if (!$stmt->execute()) {
        $stmt->close();
        throw new \RuntimeException('Failed to update education: ' . $stmt->error);
    }

    $stmt->close();

    if ($isHighest === 1) {
        pemohon_mark_highest_education($db, $applicantId, $educationId);
    } else {
        // If no entry marked as highest, ensure at least one remains (first record)
        pemohon_ensure_highest_exists($db, $applicantId);
    }

    return pemohon_find_education($userId, $educationId);
}

function pemohon_reorder_educations(int $userId, array $orderedIds): bool
{
    if (!$orderedIds) {
        return true;
    }

    $applicantId = pemohon_get_applicant_id($userId);
    if ($applicantId === null) {
        return false;
    }

    $db = pemohon_db();
    $sql = 'UPDATE applicant_educations SET display_order = ? WHERE id = ? AND applicant_id = ?';
    $stmt = $db->prepare($sql);
    if (!$stmt) {
        throw new \RuntimeException('Failed to prepare reorder statement: ' . $db->error);
    }

    $order = 1;
    foreach ($orderedIds as $educationId) {
        $educationId = (int) $educationId;
        if ($educationId <= 0) {
            continue;
        }
        $stmt->bind_param('iii', $order, $educationId, $applicantId);
        $stmt->execute();
        $order++;
    }

    $stmt->close();
    return true;
}

function pemohon_delete_education(int $userId, int $educationId): bool
{
    if ($educationId <= 0) {
        return false;
    }

    $applicantId = pemohon_get_applicant_id($userId);
    if ($applicantId === null) {
        return false;
    }

    $db = pemohon_db();
    $wasHighest = false;

    $checkStmt = $db->prepare('SELECT is_highest FROM applicant_educations WHERE id = ? AND applicant_id = ?');
    if ($checkStmt) {
        $checkStmt->bind_param('ii', $educationId, $applicantId);
        $checkStmt->execute();
        $checkStmt->bind_result($highestFlag);
        if ($checkStmt->fetch()) {
            $wasHighest = ((int) $highestFlag) === 1;
        }
        $checkStmt->close();
    }

    $sql = 'DELETE FROM applicant_educations WHERE id = ? AND applicant_id = ?';
    $stmt = $db->prepare($sql);
    if (!$stmt) {
        if ((int) $db->errno === 1146) {
            return false;
        }
        throw new \RuntimeException('Failed to prepare statement: ' . $db->error);
    }

    $stmt->bind_param('ii', $educationId, $applicantId);
    $stmt->execute();
    $affected = $stmt->affected_rows;
    $stmt->close();

    if ($affected > 0 && $wasHighest) {
        pemohon_ensure_highest_exists($db, $applicantId);
    }

    return $affected > 0;
}

function pemohon_mark_highest_education(\mysqli $db, int $applicantId, int $educationId): void
{
    $resetStmt = $db->prepare('UPDATE applicant_educations SET is_highest = 0 WHERE applicant_id = ?');
    if ($resetStmt) {
        $resetStmt->bind_param('i', $applicantId);
        $resetStmt->execute();
        $resetStmt->close();
    }

    $setStmt = $db->prepare('UPDATE applicant_educations SET is_highest = 1 WHERE id = ? AND applicant_id = ?');
    if ($setStmt) {
        $setStmt->bind_param('ii', $educationId, $applicantId);
        $setStmt->execute();
        $setStmt->close();
    }
}

function pemohon_ensure_highest_exists(\mysqli $db, int $applicantId): void
{
    $checkStmt = $db->prepare('SELECT id FROM applicant_educations WHERE applicant_id = ? AND is_highest = 1 LIMIT 1');
    if (!$checkStmt) {
        return;
    }

    $checkStmt->bind_param('i', $applicantId);
    $checkStmt->execute();
    $checkStmt->store_result();

    if ($checkStmt->num_rows === 0) {
        $checkStmt->close();
        $firstStmt = $db->prepare('SELECT id FROM applicant_educations WHERE applicant_id = ? ORDER BY display_order ASC, id ASC LIMIT 1');
        if ($firstStmt) {
            $firstStmt->bind_param('i', $applicantId);
            $firstStmt->execute();
            $firstStmt->bind_result($firstId);
            if ($firstStmt->fetch()) {
                $firstStmt->close();
                pemohon_mark_highest_education($db, $applicantId, (int) $firstId);
                return;
            }
            $firstStmt->close();
        }
    } else {
        $checkStmt->close();
    }
}
