<?php

declare(strict_types=1);

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Request;
use App\Core\Response;
use App\Core\Session;
use mysqli;
use RuntimeException;

class ApplicantApplicationsController extends Controller
{
    private mysqli $db;

    public function __construct(Request $request, Response $response, Session $session)
    {
        parent::__construct($request, $response, $session);
        $this->db = Database::connection();
    }

    public function index(): void
    {
        $applications = $this->fetchApplications();
        $statusOptions = $this->fetchStatusOptions();

        $this->view('applicants.applications', [
            'title' => 'Pengurusan Permohonan Pemohon',
            'applications' => $applications,
            'statusOptions' => $statusOptions,
            'pageScripts' => $this->pageScripts($statusOptions),
        ]);
    }

    public function update(): void
    {
        $session = $this->session;

        if (!$this->request->isAjax()) {
            $session->flash('error', 'Kaedah tidak dibenarkan.');
            $this->response->redirect('/modules/applicant/applications');
        }

        $applicationId = (int) $this->request->input('application_id', 0);
        $statusId = (int) $this->request->input('status_tauliah_id', 0);
        $note = trim((string) $this->request->input('note', ''));
        if (mb_strlen($note) > 1000) {
            $note = mb_substr($note, 0, 1000);
        }
        $token = $this->request->input('_token');

        if (!is_int($applicationId) || $applicationId <= 0) {
            $this->response->json([
                'success' => false,
                'message' => 'Permohonan tidak sah.',
            ], 422);
        }

        if (!validate_csrf_token(is_string($token) ? $token : null)) {
            $this->response->json([
                'success' => false,
                'message' => 'Sesi anda telah tamat. Sila muat semula halaman.',
            ], 419);
        }

        if ($statusId <= 0) {
            $this->response->json([
                'success' => false,
                'message' => 'Sila pilih status tauliah.',
            ], 422);
        }

        $statusOptions = $this->fetchStatusOptions();
        $selectedStatus = $statusOptions[$statusId] ?? null;

        if ($selectedStatus === null) {
            $this->response->json([
                'success' => false,
                'message' => 'Status tauliah tidak ditemui.',
            ], 422);
        }

        $application = $this->fetchApplication($applicationId);
        if ($application === null) {
            $this->response->json([
                'success' => false,
                'message' => 'Rekod permohonan tidak ditemui.',
            ], 404);
        }

        $user = $session->get('user') ?? [];
        $adminId = isset($user['id']) ? (int) $user['id'] : null;
        if ($adminId !== null && $adminId <= 0) {
            $adminId = null;
        }

        $metadata = $application['metadata_raw'];
        if (!is_array($metadata)) {
            $metadata = [];
        }

        $metadata['status_tauliah'] = [
            'id' => $selectedStatus['id'],
            'name' => $selectedStatus['name'],
            'updated_at' => date('c'),
        ];
        $metadata['admin_note'] = $note;

        $metadataJson = json_encode($metadata, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        if ($metadataJson === false) {
            $this->response->json([
                'success' => false,
                'message' => 'Tidak dapat memproses data status.',
            ], 500);
        }

        $this->db->begin_transaction();

        try {
            $stmt = $this->db->prepare(
                'UPDATE applications
                 SET current_stage = ?, metadata = ?, last_status_changed_at = NOW(), updated_by = NULLIF(?, 0), updated_at = NOW()
                 WHERE id = ?'
            );
            if (!$stmt) {
                throw new RuntimeException('Gagal menyediakan kemaskini permohonan: ' . $this->db->error);
            }
            $currentStage = $selectedStatus['name'];
            $adminIdParam = $adminId ?? 0;
            $stmt->bind_param('ssii', $currentStage, $metadataJson, $adminIdParam, $applicationId);
            $stmt->execute();
            $stmt->close();

            $this->db->commit();
        } catch (\Throwable $exception) {
            $this->db->rollback();
            $this->response->json([
                'success' => false,
                'message' => 'Kemaskini tidak berjaya: ' . $exception->getMessage(),
            ], 500);
        }

        $updatedApplication = $this->fetchApplication($applicationId);

        $this->response->json([
            'success' => true,
            'message' => 'Permohonan berjaya dikemaskini.',
            'application' => $updatedApplication,
        ]);
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    private function fetchApplications(): array
    {
        $sql = <<<SQL
            SELECT
                a.id,
                a.application_number,
                a.application_type,
                a.tauliah_type,
                a.status,
                a.current_stage,
                a.submitted_at,
                a.updated_at,
                a.metadata,
                u.name AS applicant_name,
                u.email AS applicant_email,
                u.phone AS applicant_phone,
                app.nric AS applicant_nric
            FROM applications a
            INNER JOIN applicants app ON app.id = a.applicant_id
            INNER JOIN users u ON u.id = app.user_id
            ORDER BY a.created_at DESC
        SQL;

        $result = $this->db->query($sql);
        if (!$result) {
            throw new RuntimeException('Gagal mendapatkan senarai permohonan: ' . $this->db->error);
        }

        $applications = [];
        while ($row = $result->fetch_assoc()) {
            $applications[] = $this->transformApplicationRow($row);
        }
        $result->free();

        return $applications;
    }

    /**
     * @return array<string, mixed>|null
     */
    private function fetchApplication(int $applicationId): ?array
    {
        $sql = <<<SQL
            SELECT
                a.id,
                a.application_number,
                a.application_type,
                a.tauliah_type,
                a.status,
                a.current_stage,
                a.submitted_at,
                a.updated_at,
                a.metadata,
                u.name AS applicant_name,
                u.email AS applicant_email,
                u.phone AS applicant_phone,
                app.nric AS applicant_nric
            FROM applications a
            INNER JOIN applicants app ON app.id = a.applicant_id
            INNER JOIN users u ON u.id = app.user_id
            WHERE a.id = ?
            LIMIT 1
        SQL;

        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new RuntimeException('Gagal menyediakan carian permohonan: ' . $this->db->error);
        }

        $stmt->bind_param('i', $applicationId);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc() ?: null;
        $stmt->close();

        return $row ? $this->transformApplicationRow($row) : null;
    }

    /**
     * @return array<int, array{id:int,name:string}>
     */
    private function fetchStatusOptions(): array
    {
        $sql = 'SELECT stta_id, nama FROM status_tauliah ORDER BY stta_id ASC';
        $result = $this->db->query($sql);
        if (!$result) {
            return [];
        }

        $statuses = [];
        while ($row = $result->fetch_assoc()) {
            $id = (int) $row['stta_id'];
            $name = (string) ($row['nama'] ?? '');
            if ($id <= 0 || $name === '') {
                continue;
            }
            $statuses[$id] = [
                'id' => $id,
                'name' => $name,
            ];
        }
        $result->free();

        return $statuses;
    }

    /**
     * @param array<string, mixed> $row
     *
     * @return array<string, mixed>
     */
    private function transformApplicationRow(array $row): array
    {
        $typeLabels = [
            'new' => 'Permohonan Baharu',
            'renewal' => 'Pembaharuan',
            'upgrade' => 'Naik Taraf',
            'replacement' => 'Penggantian Hilang/Rosak',
        ];

        $tauliahLabels = [
            'agama' => 'Tauliah Agama',
            'alquran' => 'Tauliah Al-Quran',
            'both' => 'Tauliah Agama & Al-Quran',
        ];

        $statusLabels = [
            'draft' => 'Draf',
            'submitted' => 'Dihantar',
            'under_review' => 'Dalam Semakan',
            'interview' => 'Temuduga',
            'approved' => 'Diluluskan',
            'rejected' => 'Ditolak',
            'kiv' => 'KIV',
            'cancelled' => 'Dibatalkan',
            'expired' => 'Tamat Tempoh',
        ];

        $metadataRaw = [];
        if (isset($row['metadata']) && $row['metadata'] !== null) {
            $decoded = json_decode((string) $row['metadata'], true);
            if (is_array($decoded)) {
                $metadataRaw = $decoded;
            }
        }

        $statusTauliah = $metadataRaw['status_tauliah'] ?? null;
        $statusTauliahId = null;
        $statusTauliahName = null;
        if (is_array($statusTauliah)) {
            $statusTauliahId = isset($statusTauliah['id']) ? (int) $statusTauliah['id'] : null;
            $statusTauliahName = isset($statusTauliah['name']) ? (string) $statusTauliah['name'] : null;
        }

        return [
            'id' => (int) $row['id'],
            'application_number' => (string) ($row['application_number'] ?? ''),
            'application_type' => (string) ($row['application_type'] ?? ''),
            'application_type_label' => $typeLabels[$row['application_type'] ?? ''] ?? ($row['application_type'] ?? ''),
            'tauliah_type' => (string) ($row['tauliah_type'] ?? ''),
            'tauliah_type_label' => $tauliahLabels[$row['tauliah_type'] ?? ''] ?? ($row['tauliah_type'] ?? ''),
            'status' => (string) ($row['status'] ?? ''),
            'status_label' => $statusLabels[$row['status'] ?? ''] ?? ($row['status'] ?? ''),
            'current_stage' => (string) ($row['current_stage'] ?? ''),
            'submitted_at' => $this->formatDateTime($row['submitted_at'] ?? null, 'Y-m-d'),
            'submitted_at_display' => $this->formatDateTime($row['submitted_at'] ?? null, 'd/m/Y'),
            'updated_at' => $this->formatDateTime($row['updated_at'] ?? null, 'Y-m-d H:i:s'),
            'updated_at_display' => $this->formatDateTime($row['updated_at'] ?? null, 'd/m/Y H:i'),
            'applicant_name' => (string) ($row['applicant_name'] ?? ''),
            'applicant_email' => (string) ($row['applicant_email'] ?? ''),
            'applicant_phone' => (string) ($row['applicant_phone'] ?? ''),
            'applicant_nric' => (string) ($row['applicant_nric'] ?? ''),
            'status_tauliah_id' => $statusTauliahId,
            'status_tauliah_name' => $statusTauliahName,
            'admin_note' => isset($metadataRaw['admin_note']) ? (string) $metadataRaw['admin_note'] : '',
            'metadata_raw' => $metadataRaw,
        ];
    }

    private function formatDateTime(mixed $value, string $format): ?string
    {
        if ($value === null || $value === '') {
            return null;
        }

        $timestamp = strtotime((string) $value);
        if ($timestamp === false) {
            return null;
        }

        return date($format, $timestamp);
    }

    /**
     * @param array<int, array{id:int,name:string}> $statusOptions
     */
    private function pageScripts(array $statusOptions): string
    {
        $statusOptionsJson = json_encode(array_values($statusOptions), JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

        return <<<SCRIPT
<script>
document.addEventListener('DOMContentLoaded', function () {
    var statusOptions = {$statusOptionsJson};
    var table = document.getElementById('applications-table');
    var modalEl = document.getElementById('applicationModal');
    var form = document.getElementById('application-update-form');
    var statusSelect = document.getElementById('status-tauliah');
    var noteInput = document.getElementById('status-note');
    var applicationIdInput = document.getElementById('application-id');
    var feedback = document.getElementById('application-update-feedback');
    var submitBtn = form ? form.querySelector('button[type="submit"]') : null;
    var modalInstance = null;
    var modalJQuery = null;
    var dataTableInstance = null;

    if (window.jQuery && window.jQuery.fn && typeof window.jQuery.fn.DataTable === 'function' && table) {
        var jqTable = window.jQuery(table);
        if (window.jQuery.fn.DataTable.isDataTable(jqTable)) {
            dataTableInstance = jqTable.DataTable();
        } else {
            dataTableInstance = jqTable.DataTable({
                responsive: true,
                order: [[6, 'desc']],
                pageLength: 10,
                columnDefs: [
                    { targets: [7], orderable: false, searchable: false }
                ]
            });
        }
    }

    if (window.bootstrap && window.bootstrap.Modal && modalEl) {
        modalInstance = new window.bootstrap.Modal(modalEl, {backdrop: 'static'});
    } else if (window.jQuery && typeof window.jQuery.fn.modal === 'function' && modalEl) {
        modalJQuery = window.jQuery(modalEl);
    }

    function openModal() {
        if (modalInstance) {
            modalInstance.show();
            return;
        }
        if (modalJQuery) {
            modalJQuery.modal({backdrop: 'static', keyboard: false}).modal('show');
            return;
        }
        if (modalEl) {
            modalEl.classList.add('show');
            modalEl.style.display = 'block';
        }
    }

    function closeModal() {
        if (modalInstance) {
            modalInstance.hide();
            return;
        }
        if (modalJQuery) {
            modalJQuery.modal('hide');
            return;
        }
        if (modalEl) {
            modalEl.classList.remove('show');
            modalEl.style.display = 'none';
        }
    }

    window.etauliahApplicantApplications = window.etauliahApplicantApplications || {};
    window.etauliahApplicantApplications.openModal = openModal;
    window.etauliahApplicantApplications.closeModal = closeModal;

    if (statusSelect && statusOptions) {
        statusSelect.innerHTML = '<option value="">-- Pilih Status Tauliah --</option>';
        statusOptions.forEach(function (option) {
            var opt = document.createElement('option');
            opt.value = option.id;
            opt.textContent = option.name;
            statusSelect.appendChild(opt);
        });
    }

    function clearFeedback() {
        if (!feedback) {
            return;
        }
        feedback.classList.add('d-none');
        feedback.classList.remove('alert-success', 'alert-danger');
        feedback.textContent = '';
    }

    function showFeedback(type, message) {
        if (!feedback) {
            return;
        }
        feedback.classList.remove('d-none', 'alert-success', 'alert-danger');
        feedback.classList.add(type === 'success' ? 'alert-success' : 'alert-danger');
        feedback.textContent = message || '';
    }

    function toggleSubmitting(isSubmitting) {
        if (!submitBtn) {
            return;
        }
        if (isSubmitting) {
            submitBtn.disabled = true;
            if (!submitBtn.dataset.originalText) {
                submitBtn.dataset.originalText = submitBtn.textContent;
            }
            submitBtn.textContent = 'Mengemaskini...';
        } else {
            submitBtn.disabled = false;
            if (submitBtn.dataset.originalText) {
                submitBtn.textContent = submitBtn.dataset.originalText;
            }
        }
    }

    function findRow(applicationId) {
        if (!table) {
            return null;
        }
        var rows = table.querySelectorAll('tbody tr');
        for (var i = 0; i < rows.length; i++) {
            var row = rows[i];
            if (parseInt(row.getAttribute('data-application-id') || '0', 10) === applicationId) {
                return row;
            }
        }
        return null;
    }

    function updateRow(row, application) {
        if (!row || !application) {
            return;
        }
        row.setAttribute('data-application-json', JSON.stringify(application));

        var cells = row.querySelectorAll('td');
        if (cells.length >= 7) {
            cells[0].textContent = application.application_number || '-';
            cells[1].textContent = application.applicant_name || '-';
            cells[2].textContent = application.application_type_label || '-';
            cells[3].textContent = application.tauliah_type_label || '-';
            cells[4].textContent = application.status_tauliah_name || application.current_stage || '-';
            cells[5].textContent = application.status_label || '-';
            cells[6].textContent = application.updated_at_display || '-';
        }
    }

    function handleEditClick(event) {
        event.preventDefault();
        var trigger = event.currentTarget || event.target;
        if (!trigger) {
            return;
        }

        var rowNode = trigger.closest ? trigger.closest('tr') : null;

        if (window.jQuery && dataTableInstance) {
            var jqTrigger = window.jQuery(trigger);
            var jqRow = jqTrigger.closest('tr');
            if (jqRow.length && jqRow.hasClass('child') && jqRow.prev().length) {
                jqRow = jqRow.prev();
            }
            if (jqRow.length) {
                var dtRow = dataTableInstance.row(jqRow);
                if (dtRow && typeof dtRow.node === 'function') {
                    rowNode = dtRow.node();
                }
            }
        }

        if (!rowNode && trigger.closest) {
            rowNode = trigger.closest('tr');
        }
        if (!rowNode) {
            return;
        }

        var applicationJson = rowNode.getAttribute('data-application-json');
        if (!applicationJson && window.jQuery) {
            applicationJson = window.jQuery(rowNode).attr('data-application-json') || null;
        }
        if (!applicationJson) {
            return;
        }

        try {
            var application = JSON.parse(applicationJson);
            clearFeedback();
            if (applicationIdInput) {
                applicationIdInput.value = application.id || '';
            }
            if (statusSelect) {
                statusSelect.value = application.status_tauliah_id || '';
            }
            if (noteInput) {
                noteInput.value = application.admin_note || '';
            }
            openModal();
        } catch (error) {
            console.error('Gagal memuatkan data permohonan:', error);
        }
    }

    if (window.jQuery && dataTableInstance) {
        window.jQuery(table).on('click', '[data-action=\"edit-application\"]', handleEditClick);
    } else {
        document.addEventListener('click', function (event) {
            var trigger = event.target.closest('[data-action=\"edit-application\"]');
            if (!trigger) {
                return;
            }
            handleEditClick(event);
        });
    }

    if (form) {
        form.addEventListener('submit', function (event) {
            event.preventDefault();
            clearFeedback();

            var applicationId = parseInt(applicationIdInput ? applicationIdInput.value : '0', 10);
            if (!applicationId) {
                showFeedback('danger', 'Rekod permohonan tidak sah.');
                return;
            }

            var formData = new FormData(form);
            toggleSubmitting(true);

            fetch(form.action, {
                method: 'POST',
                body: formData,
                credentials: 'same-origin',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            }).then(function (response) {
                return response.json().catch(function () { return {}; }).then(function (data) {
                    return { ok: response.ok, status: response.status, data: data };
                });
            }).then(function (payload) {
                var data = payload.data || {};
                if (!payload.ok || data.success === false) {
                    showFeedback('danger', data.message || 'Kemaskini tidak dapat diselesaikan.');
                    return;
                }

                showFeedback('success', data.message || 'Permohonan berjaya dikemaskini.');
                if (data.application) {
                    var row = findRow(data.application.id);
                    updateRow(row, data.application);
                }
                setTimeout(function () {
                    closeModal();
                }, 800);
            }).catch(function (error) {
                console.error('Ralat kemaskini permohonan:', error);
                showFeedback('danger', 'Sistem tidak dapat memproses permintaan.');
            }).finally(function () {
                toggleSubmitting(false);
            });
        });
    }
});
</script>
SCRIPT;
    }
}
