<?php

declare(strict_types=1);

namespace App\Services;

use PHPMailer\PHPMailer\Exception as MailException;
use PHPMailer\PHPMailer\PHPMailer;
use RuntimeException;

class MailerService
{
    /** @var array<string, mixed> */
    private array $config;

    public function __construct()
    {
        $config = config('mail');
        if (!is_array($config) || $config === []) {
            throw new RuntimeException('Mail configuration is missing.');
        }

        $this->config = $config;
    }

    public function send(string $toEmail, string $toName, string $subject, string $htmlBody, ?string $textBody = null): void
    {
        $mail = new PHPMailer(true);
        $config = $this->config;

        try {
            $mail->isSMTP();
            $mail->Host = (string) ($config['host'] ?? '');
            $mail->Port = (int) ($config['port'] ?? 587);
            $mail->SMTPAuth = true;
            $mail->Username = (string) ($config['username'] ?? '');
            $mail->Password = (string) ($config['password'] ?? '');
            $mail->Timeout = (int) ($config['timeout'] ?? 10);
            $mail->CharSet = 'UTF-8';

            $encryption = strtolower((string) ($config['encryption'] ?? ''));
            if ($encryption === 'tls') {
                $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
            } elseif ($encryption === 'ssl' || $encryption === 'smtps') {
                $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
            }

            $fromAddress = (string) ($config['from_address'] ?? $config['username'] ?? '');
            $fromName = (string) ($config['from_name'] ?? 'System');

            if ($fromAddress === '') {
                throw new RuntimeException('From address is not configured.');
            }

            $mail->setFrom($fromAddress, $fromName);

            $replyAddress = (string) ($config['reply_to_address'] ?? '');
            if ($replyAddress !== '') {
                $mail->addReplyTo(
                    $replyAddress,
                    (string) ($config['reply_to_name'] ?? $fromName)
                );
            }

            $mail->addAddress($toEmail, $toName);
            $mail->isHTML(true);
            $mail->Subject = $subject;
            $mail->Body = $htmlBody;
            $mail->AltBody = $textBody ?? strip_tags($htmlBody);

            $mail->send();
        } catch (MailException $exception) {
            throw new RuntimeException('Failed to send email: ' . $exception->getMessage(), 0, $exception);
        }
    }
}
