<?php

declare(strict_types=1);

namespace App\Services;

use App\Core\Database;
use App\Support\UserRepository;
use DateInterval;
use DateTimeImmutable;

class PasswordResetService
{
    private const EXPIRATION_MINUTES = 60;

    private UserRepository $users;
    private MailerService $mailer;

    public function __construct()
    {
        $this->users = new UserRepository();
        $this->mailer = new MailerService();
    }

    public function requestTemporaryPassword(string $email): bool
    {
        $user = $this->users->findByEmail($email);
        if (!$user) {
            return false;
        }

        $temporaryPassword = $this->generateTemporaryPassword();
        $hashedToken = password_hash($temporaryPassword, PASSWORD_DEFAULT);

        $this->storeTemporaryPassword($email, $hashedToken);

        $name = (string) ($user['name'] ?? 'Pengguna eTauliah');
        $resetUrl = route('/reset_password.php');

        $subject = 'Tetapan Semula Kata Laluan eTauliah';
        $htmlBody = $this->buildHtmlEmail($name, $temporaryPassword, $resetUrl);
        $textBody = $this->buildTextEmail($name, $temporaryPassword, $resetUrl);

        $this->mailer->send($email, $name, $subject, $htmlBody, $textBody);
        return true;
    }

    public function resetPassword(string $email, string $temporaryPassword, string $newPassword): ?array
    {
        $user = $this->users->findByEmail($email);
        if (!$user) {
            return null;
        }

        $resetRecord = $this->findResetRecord($email);
        if (!$resetRecord) {
            return null;
        }

        if (!$this->isResetValid($resetRecord, $temporaryPassword)) {
            return null;
        }

        $hashed = password_hash($newPassword, PASSWORD_DEFAULT);
        $this->updateUserPassword((int) $user['id'], $hashed);
        $this->deleteResetRecord($email);

        unset($user['password']);
        return $user;
    }

    private function generateTemporaryPassword(): string
    {
        $alphabet = 'ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz23456789@#%&!?';
        $length = 12;
        $maxIndex = strlen($alphabet) - 1;
        $password = '';

        for ($i = 0; $i < $length; $i++) {
            $index = random_int(0, $maxIndex);
            $password .= $alphabet[$index];
        }

        return $password;
    }

    private function storeTemporaryPassword(string $email, string $hashedToken): void
    {
        $sql = <<<'SQL'
INSERT INTO password_resets (email, token, created_at)
VALUES (?, ?, NOW())
ON DUPLICATE KEY UPDATE token = VALUES(token), created_at = VALUES(created_at)
SQL;
        Database::query($sql, 'ss', [$email, $hashedToken]);
    }

    /**
     * @return array<string, mixed>|null
     */
    private function findResetRecord(string $email): ?array
    {
        $sql = 'SELECT token, created_at FROM password_resets WHERE email = ? LIMIT 1';
        $result = Database::query($sql, 's', [$email]);
        $row = $result->fetch_assoc();
        $result->free();

        return $row ?: null;
    }

    /**
     * @param array<string, mixed> $record
     */
    private function isResetValid(array $record, string $temporaryPassword): bool
    {
        $createdAt = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', (string) $record['created_at']);
        if (!$createdAt) {
            return false;
        }

        $expiresAt = $createdAt->add(new DateInterval('PT' . self::EXPIRATION_MINUTES . 'M'));
        if ($expiresAt < new DateTimeImmutable('now')) {
            return false;
        }

        $token = (string) $record['token'];
        return password_verify($temporaryPassword, $token);
    }

    private function updateUserPassword(int $userId, string $hashedPassword): void
    {
        $sql = 'UPDATE users SET password = ?, force_password_change = 0, updated_at = CURRENT_TIMESTAMP WHERE id = ?';
        Database::query($sql, 'si', [$hashedPassword, $userId]);
    }

    private function deleteResetRecord(string $email): void
    {
        Database::query('DELETE FROM password_resets WHERE email = ?', 's', [$email]);
    }

    private function buildHtmlEmail(string $name, string $temporaryPassword, string $resetUrl): string
    {
        $escapedName = htmlspecialchars($name, ENT_QUOTES, 'UTF-8');
        $escapedTemp = htmlspecialchars($temporaryPassword, ENT_QUOTES, 'UTF-8');
        $escapedUrl = htmlspecialchars($resetUrl, ENT_QUOTES, 'UTF-8');

        return <<<HTML
<p>Hai {$escapedName},</p>
<p>Kami menerima permintaan untuk menetapkan semula kata laluan bagi akaun eTauliah anda.</p>
<p><strong>Kata Laluan Sementara:</strong> {$escapedTemp}</p>
<p>Sila klik pautan di bawah untuk menetapkan kata laluan baharu anda:</p>
<p><a href="{$escapedUrl}" target="_blank" rel="noopener">Tetapkan Kata Laluan Baharu</a></p>
<p>Kata laluan sementara ini sah selama 60 minit sahaja. Jika anda tidak membuat permintaan ini, sila abaikan emel ini.</p>
<p>Terima kasih,<br>Pasukan eTauliah</p>
HTML;
    }

    private function buildTextEmail(string $name, string $temporaryPassword, string $resetUrl): string
    {
        return <<<TEXT
Hai {$name},

Kami menerima permintaan untuk menetapkan semula kata laluan bagi akaun eTauliah anda.

Kata Laluan Sementara: {$temporaryPassword}

Tetapkan kata laluan baharu anda di: {$resetUrl}

Kata laluan sementara ini sah selama 60 minit sahaja. Jika anda tidak membuat permintaan ini, sila abaikan emel ini.

Terima kasih,
Pasukan eTauliah
TEXT;
    }
}
