<?php

declare(strict_types=1);

require_once __DIR__ . '/../includes/helpers.php';
require_once __DIR__ . '/../includes/auth.php';

pemohon_require_login();

$currentUser = pemohon_current_user();
if ($currentUser === null) {
    pemohon_redirect('/pemohon');
}

$db = pemohon_db();
$userId = (int) $currentUser['id'];
$isAjax = isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strcasecmp((string) $_SERVER['HTTP_X_REQUESTED_WITH'], 'xmlhttprequest') === 0;

function pemohon_generate_application_number(mysqli $db): string
{
    $year = date('Y');
    $prefix = sprintf('ETA-%s-', $year);

    $sql = 'SELECT application_number
            FROM applications
            WHERE application_number LIKE CONCAT(?, "%")
            ORDER BY application_number DESC
            LIMIT 1 FOR UPDATE';
    $stmt = $db->prepare($sql);
    if (!$stmt) {
        throw new RuntimeException('Gagal menyediakan penjana nombor permohonan: ' . $db->error);
    }

    $stmt->bind_param('s', $prefix);
    $stmt->execute();
    $stmt->bind_result($latestNumber);
    $stmt->fetch();
    $stmt->close();

    if (is_string($latestNumber) && preg_match('/(\d+)$/', $latestNumber, $matches)) {
        $sequence = (int) $matches[1] + 1;
    } else {
        $sequence = 1;
    }

    return $prefix . str_pad((string) $sequence, 4, '0', STR_PAD_LEFT);
}

$applicationTypes = [
    'new' => 'Permohonan Baharu',
    'renewal' => 'Pembaharuan Tempoh',
    'upgrade' => 'Naik Taraf Kategori',
    'replacement' => 'Penggantian Hilang/Rosak',
];

$tauliahTypes = [
    'agama' => 'Tauliah Agama',
    'alquran' => 'Tauliah Al-Quran',
    'both' => 'Tauliah Agama & Al-Quran',
];

$categories = [];
$categoryStmt = $db->prepare('SELECT kata_id, nama, deskripsi FROM kategori_tauliah ORDER BY kata_id');
if ($categoryStmt) {
    $categoryStmt->execute();
    $categoryResult = $categoryStmt->get_result();
    while ($row = $categoryResult->fetch_assoc()) {
        $categories[(int) $row['kata_id']] = $row;
    }
    $categoryStmt->close();
}

$states = [];
$stateStmt = $db->prepare('SELECT negeri_id, nama FROM negeri ORDER BY nama');
if ($stateStmt) {
    $stateStmt->execute();
    $stateResult = $stateStmt->get_result();
    while ($row = $stateResult->fetch_assoc()) {
        $states[(int) $row['negeri_id']] = $row['nama'];
    }
    $stateStmt->close();
}

$districts = [];
$districtStmt = $db->prepare('SELECT daerah_id, negeri_id, nama FROM daerah ORDER BY negeri_id, nama');
if ($districtStmt) {
    $districtStmt->execute();
    $districtResult = $districtStmt->get_result();
    while ($row = $districtResult->fetch_assoc()) {
        $districts[(int) $row['daerah_id']] = [
            'name' => $row['nama'],
            'state_id' => (int) $row['negeri_id'],
        ];
    }
    $districtStmt->close();
}

$districtsByState = [];
foreach ($districts as $districtId => $district) {
    $stateId = $district['state_id'];
    if (!isset($districtsByState[$stateId])) {
        $districtsByState[$stateId] = [];
    }
    $districtsByState[$stateId][] = [
        'id' => $districtId,
        'name' => $district['name'],
    ];
}

$melakaStateId = null;
foreach ($states as $id => $name) {
    if (strcasecmp((string) $name, 'Melaka') === 0) {
        $melakaStateId = $id;
        break;
    }
}

if ($melakaStateId !== null) {
    $states = [$melakaStateId => $states[$melakaStateId]];
    foreach ($districts as $districtId => $district) {
        if ($district['state_id'] !== $melakaStateId) {
            unset($districts[$districtId]);
        }
    }
    $districtsByState = isset($districtsByState[$melakaStateId])
        ? [$melakaStateId => $districtsByState[$melakaStateId]]
        : [$melakaStateId => []];
} else {
    $melakaStateId = array_key_first($states);
}
$defaultStateId = $melakaStateId !== null ? (string) $melakaStateId : '';

$announcementSql = <<<'SQL'
    SELECT a.title, a.body, COALESCE(a.published_from, a.created_at) AS published_at
    FROM announcements a
    LEFT JOIN announcement_targets t ON t.announcement_id = a.id
    WHERE a.is_active = 1
      AND (
            a.audience IN ('all', 'applicant')
            OR (a.audience = 'custom' AND t.target_role = 'applicant')
          )
      AND (a.published_from IS NULL OR a.published_from <= NOW())
      AND (a.published_until IS NULL OR a.published_until >= NOW())
    GROUP BY a.id
    ORDER BY published_at DESC
    LIMIT 5
SQL;
$announcements = [];
$announcementResult = $db->query($announcementSql);
if ($announcementResult instanceof mysqli_result) {
    while ($row = $announcementResult->fetch_assoc()) {
        $announcements[] = $row;
    }
    $announcementResult->free();
}

$applicant = null;
$applicantStmt = $db->prepare('SELECT id, nric FROM applicants WHERE user_id = ? LIMIT 1');
if ($applicantStmt) {
    $applicantStmt->bind_param('i', $userId);
    $applicantStmt->execute();
    $applicantResult = $applicantStmt->get_result();
    $applicant = $applicantResult->fetch_assoc() ?: null;
    $applicantStmt->close();
}

$currentProfile = pemohon_fetch_profile($userId) ?? [];

$formData = [
    'nric' => $currentProfile['nric'] ?? '',
    'application_type' => 'new',
    'tauliah_type' => 'agama',
    'tauliah_category' => '',
    'state_id' => $defaultStateId,
    'district_id' => '',
    'teaching_location_detail' => '',
    'experience_years' => '',
    'subjects' => '',
    'notes' => '',
];

if ($applicant && isset($applicant['nric']) && $applicant['nric'] !== '') {
    $formData['nric'] = (string) $applicant['nric'];
}

$errors = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $csrfToken = $_POST['_token'] ?? '';
    if (!pemohon_validate_csrf_token($csrfToken)) {
        $errors['_global'] = 'Sesi permohonan tamat. Sila cuba sekali lagi.';
    } else {
        $formData['nric'] = preg_replace('/\D+/', '', (string) ($_POST['nric'] ?? ''));
        $formData['application_type'] = (string) ($_POST['application_type'] ?? '');
        $formData['tauliah_type'] = (string) ($_POST['tauliah_type'] ?? '');
        $formData['tauliah_category'] = (string) ($_POST['tauliah_category'] ?? '');
        $formData['state_id'] = (string) ($_POST['state_id'] ?? '');
        $formData['district_id'] = (string) ($_POST['district_id'] ?? '');
        $formData['teaching_location_detail'] = trim((string) ($_POST['teaching_location_detail'] ?? ''));
        $formData['experience_years'] = trim((string) ($_POST['experience_years'] ?? ''));
        $formData['subjects'] = trim((string) ($_POST['subjects'] ?? ''));
        $formData['notes'] = trim((string) ($_POST['notes'] ?? ''));

        if (!preg_match('/^\d{12}$/', $formData['nric'])) {
            $errors['nric'] = 'No. Kad Pengenalan hendaklah mempunyai 12 digit tanpa sengkang.';
        }

        if (!array_key_exists($formData['application_type'], $applicationTypes)) {
            $errors['application_type'] = 'Jenis permohonan tidak sah.';
        }

        if (!array_key_exists($formData['tauliah_type'], $tauliahTypes)) {
            $errors['tauliah_type'] = 'Jenis tauliah tidak sah.';
        }

        $categoryId = (int) $formData['tauliah_category'];
        if (!isset($categories[$categoryId])) {
            $errors['tauliah_category'] = 'Sila pilih kategori tauliah yang sah.';
        }

        $stateId = (int) $formData['state_id'];
        if ($stateId <= 0 || !isset($states[$stateId])) {
            $errors['state_id'] = 'Sila pilih negeri mengajar.';
        }

        $districtId = (int) $formData['district_id'];
        if ($districtId > 0 && (!isset($districts[$districtId]) || $districts[$districtId]['state_id'] !== $stateId)) {
            $errors['district_id'] = 'Daerah yang dipilih tidak sepadan dengan negeri.';
        }

        if ($formData['experience_years'] !== '' && !preg_match('/^\d{1,2}$/', $formData['experience_years'])) {
            $errors['experience_years'] = 'Sila masukkan tempoh pengalaman yang sah dalam tahun.';
        }

        if (!$errors) {
            $db->begin_transaction();
            try {
                $uniqueCheckStmt = $db->prepare('SELECT id FROM users WHERE nric = ? AND id <> ? LIMIT 1');
                if (!$uniqueCheckStmt) {
                    throw new RuntimeException('Gagal memeriksa No. Kad Pengenalan: ' . $db->error);
                }
                $uniqueCheckStmt->bind_param('si', $formData['nric'], $userId);
                $uniqueCheckStmt->execute();
                $uniqueCheckStmt->store_result();
                if ($uniqueCheckStmt->num_rows > 0) {
                    $errors['nric'] = 'No. Kad Pengenalan telah digunakan oleh akaun lain.';
                }
                $uniqueCheckStmt->close();

                if ($errors) {
                    $db->rollback();
                } else {
                    $applicantId = $applicant['id'] ?? null;
                    if ($applicantId === null) {
                        $insertApplicant = $db->prepare('INSERT INTO applicants (user_id, nric, created_at, updated_at) VALUES (?, NULLIF(?, \'\'), NOW(), NOW())');
                        if (!$insertApplicant) {
                            throw new RuntimeException('Gagal mendaftar pemohon: ' . $db->error);
                        }
                        $insertApplicant->bind_param('is', $userId, $formData['nric']);
                        $insertApplicant->execute();
                        $applicantId = (int) $db->insert_id;
                        $insertApplicant->close();
                    } else {
                        $applicantId = (int) $applicantId;
                        $updateApplicant = $db->prepare('UPDATE applicants SET nric = ? WHERE id = ?');
                        if (!$updateApplicant) {
                            throw new RuntimeException('Gagal mengemaskini maklumat pemohon: ' . $db->error);
                        }
                        $updateApplicant->bind_param('si', $formData['nric'], $applicantId);
                        $updateApplicant->execute();
                        $updateApplicant->close();
                    }

                    $updateUser = $db->prepare('UPDATE users SET nric = ? WHERE id = ?');
                    if (!$updateUser) {
                        throw new RuntimeException('Gagal mengemaskini maklumat pengguna: ' . $db->error);
                    }
                    $updateUser->bind_param('si', $formData['nric'], $userId);
                    $updateUser->execute();
                    $updateUser->close();

                    $dupStatus = ['draft', 'submitted', 'under_review', 'interview', 'kiv'];
                    $placeholders = implode(',', array_fill(0, count($dupStatus), '?'));
                    $dupQuery = sprintf(
                        'SELECT COUNT(*) FROM applications WHERE applicant_id = ? AND application_type = ? AND status IN (%s)',
                        $placeholders
                    );
                    $dupStmt = $db->prepare($dupQuery);
                    if (!$dupStmt) {
                        throw new RuntimeException('Gagal menyemak permohonan sedia ada: ' . $db->error);
                    }

                    $bindTypes = 'is' . str_repeat('s', count($dupStatus));
                    $dupParams = array_merge([$bindTypes, $applicantId, $formData['application_type']], $dupStatus);
                    $dupStmt->bind_param(...$dupParams);
                    $dupStmt->execute();
                    $dupStmt->bind_result($existingCount);
                    $dupStmt->fetch();
                    $dupStmt->close();

                    if ((int) $existingCount > 0) {
                        $errors['_global'] = 'Permohonan bagi jenis tersebut masih dalam proses. Sila semak status permohonan anda.';
                        $db->rollback();
                    } else {
                        $applicationNumber = pemohon_generate_application_number($db);
                        $submittedAt = date('Y-m-d H:i:s');
                        $experienceYears = $formData['experience_years'] === '' ? null : (int) $formData['experience_years'];

                        $metadata = [
                            'category' => [
                                'id' => $categoryId,
                                'name' => $categories[$categoryId]['nama'] ?? null,
                                'description' => $categories[$categoryId]['deskripsi'] ?? null,
                            ],
                            'teaching_location' => [
                                'state' => [
                                    'id' => $stateId,
                                    'name' => $states[$stateId] ?? null,
                                ],
                                'district' => $districtId > 0 ? [
                                    'id' => $districtId,
                                    'name' => $districts[$districtId]['name'] ?? null,
                                ] : null,
                                'detail' => $formData['teaching_location_detail'],
                            ],
                            'experience_years' => $experienceYears,
                            'subjects' => $formData['subjects'],
                            'notes' => $formData['notes'],
                        ];

                        $metadataJson = json_encode($metadata, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

                        $insertSql = <<<'SQL'
                            INSERT INTO applications
                                (applicant_id, application_number, application_type, tauliah_type, status,
                                 current_stage, submitted_at, created_by, updated_by, metadata, created_at, updated_at)
                            VALUES
                                (?, ?, ?, ?, 'submitted',
                                 'semakan_awal', ?, ?, ?, ?, NOW(), NOW())
SQL;
                        $insertStmt = $db->prepare($insertSql);
                        if (!$insertStmt) {
                            throw new RuntimeException('Gagal menyimpan permohonan: ' . $db->error);
                        }

                        $insertStmt->bind_param(
                            'issssiis',
                            $applicantId,
                            $applicationNumber,
                            $formData['application_type'],
                            $formData['tauliah_type'],
                            $submittedAt,
                            $userId,
                            $userId,
                            $metadataJson
                        );
                        $insertStmt->execute();
                        $insertStmt->close();

                        $db->commit();

                        $successMessage = sprintf(
                            'Permohonan %s berjaya dihantar. Nombor rujukan anda ialah %s.',
                            strtolower($applicationTypes[$formData['application_type']] ?? 'baharu'),
                            $applicationNumber
                        );

                        $_SESSION['pemohon_message'] = $successMessage;

                        if ($isAjax) {
                            header('Content-Type: application/json; charset=utf-8');
                            echo json_encode([
                                'success' => true,
                                'message' => $successMessage,
                                'redirect' => '/pemohon/applications/index.php',
                                'application_number' => $applicationNumber,
                            ]);
                            exit;
                        }

                        pemohon_redirect('/pemohon/applications/index.php');
                    }
                }
            } catch (Throwable $exception) {
                $db->rollback();
                $errors['_global'] = 'Permohonan tidak dapat dihantar buat masa ini. Sila cuba lagi atau hubungi pentadbir.';
                error_log('Permohonan pemohon gagal: ' . $exception->getMessage());
            }
        }
    }
}

if ($isAjax && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $statusCode = !empty($errors['_global']) ? 500 : 422;
    http_response_code($statusCode);
    header('Content-Type: application/json; charset=utf-8');
    $message = $errors['_global'] ?? 'Sila semak ruangan yang ditandakan.';
    echo json_encode([
        'success' => false,
        'message' => $message,
        'errors' => $errors,
    ]);
    exit;
}

$districtsJson = json_encode($districtsByState, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

$pageTitle = 'Permohonan Baharu - eTauliah';

include __DIR__ . '/../header.php';
include __DIR__ . '/../sidebar.php';
?>
        <div class="page-wrapper">
            <div class="container-fluid">
                <div class="row page-titles">
                    <div class="col-md-6 col-12 align-self-center">
                        <h3 class="text-themecolor">Permohonan Baharu</h3>
                        <ol class="breadcrumb">
                            <li class="breadcrumb-item"><a href="/pemohon/profile.php">Utama</a></li>
                            <li class="breadcrumb-item"><a href="/pemohon/applications/index.php">Permohonan Saya</a></li>
                            <li class="breadcrumb-item active">Permohonan Baharu</li>
                        </ol>
                    </div>
                    <div class="col-md-6 col-12 align-self-center text-end">
                        <a href="/pemohon/applications/index.php" class="btn btn-outline-secondary">
                            <i class="ti-arrow-left"></i> Kembali
                        </a>
                    </div>
                </div>

                <div class="row">
                    <div class="col-lg-8 col-md-7 col-12">
                        <div class="card">
                            <div class="card-body">
                                <h4 class="card-title mb-4">Maklumat Permohonan</h4>
                                <div id="application-alert" class="alert <?php echo isset($errors['_global']) ? 'alert-danger' : 'd-none'; ?>" role="alert">
                                    <?php echo isset($errors['_global']) ? htmlspecialchars($errors['_global'], ENT_QUOTES, 'UTF-8') : ''; ?>
                                </div>
                                <form method="POST" action="/pemohon/applications/create.php" novalidate data-role="application-form">
                                    <?php echo pemohon_csrf_field(); ?>
                                    <div class="mb-3">
                                        <label for="nric" class="form-label">No. Kad Pengenalan</label>
                                        <input type="text" class="form-control<?php echo isset($errors['nric']) ? ' is-invalid' : ''; ?>"
                                               id="nric" name="nric" maxlength="12" inputmode="numeric"
                                               placeholder="Contoh: 900101015555"
                                               value="<?php echo htmlspecialchars($formData['nric'], ENT_QUOTES, 'UTF-8'); ?>">
                                        <small class="form-text text-muted<?php echo isset($errors['nric']) ? ' d-none' : ''; ?>" data-helper-for="nric">Sila masukkan tanpa sengkang. Sistem akan menyemak kewujudan permohonan sedia ada.</small>
                                        <div class="invalid-feedback d-block<?php echo isset($errors['nric']) ? '' : ' d-none'; ?>" data-error-for="nric">
                                            <?php echo isset($errors['nric']) ? htmlspecialchars($errors['nric'], ENT_QUOTES, 'UTF-8') : ''; ?>
                                        </div>
                                    </div>
                                    <div class="mb-3">
                                        <label for="application_type" class="form-label">Jenis Permohonan</label>
                                        <select class="form-select<?php echo isset($errors['application_type']) ? ' is-invalid' : ''; ?>"
                                                id="application_type" name="application_type" required>
                                            <?php foreach ($applicationTypes as $value => $label): ?>
                                                <option value="<?php echo htmlspecialchars($value, ENT_QUOTES, 'UTF-8'); ?>"
                                                    <?php echo $formData['application_type'] === $value ? 'selected' : ''; ?>>
                                                    <?php echo htmlspecialchars($label, ENT_QUOTES, 'UTF-8'); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                        <div class="invalid-feedback d-block<?php echo isset($errors['application_type']) ? '' : ' d-none'; ?>" data-error-for="application_type">
                                            <?php echo isset($errors['application_type']) ? htmlspecialchars($errors['application_type'], ENT_QUOTES, 'UTF-8') : ''; ?>
                                        </div>
                                    </div>
                                    <div class="mb-3">
                                        <label for="tauliah_type" class="form-label">Jenis Tauliah Dipohon</label>
                                        <select class="form-select<?php echo isset($errors['tauliah_type']) ? ' is-invalid' : ''; ?>"
                                                id="tauliah_type" name="tauliah_type" required>
                                            <?php foreach ($tauliahTypes as $value => $label): ?>
                                                <option value="<?php echo htmlspecialchars($value, ENT_QUOTES, 'UTF-8'); ?>"
                                                    <?php echo $formData['tauliah_type'] === $value ? 'selected' : ''; ?>>
                                                    <?php echo htmlspecialchars($label, ENT_QUOTES, 'UTF-8'); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                        <div class="invalid-feedback d-block<?php echo isset($errors['tauliah_type']) ? '' : ' d-none'; ?>" data-error-for="tauliah_type">
                                            <?php echo isset($errors['tauliah_type']) ? htmlspecialchars($errors['tauliah_type'], ENT_QUOTES, 'UTF-8') : ''; ?>
                                        </div>
                                    </div>
                                    <div class="mb-3">
                                        <label for="tauliah_category" class="form-label">Kategori Tauliah</label>
                                        <select class="form-select<?php echo isset($errors['tauliah_category']) ? ' is-invalid' : ''; ?>"
                                                id="tauliah_category" name="tauliah_category" required>
                                            <option value="">-- Pilih Kategori --</option>
                                            <?php foreach ($categories as $id => $category): ?>
                                                <option value="<?php echo $id; ?>"
                                                    <?php echo (int) $formData['tauliah_category'] === $id ? 'selected' : ''; ?>>
                                                    <?php echo htmlspecialchars($category['nama'], ENT_QUOTES, 'UTF-8'); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                        <small class="form-text text-muted<?php echo isset($errors['tauliah_category']) ? ' d-none' : ''; ?>" data-helper-for="tauliah_category">Kategori menentukan kawasan kebenaran mengajar.</small>
                                        <div class="invalid-feedback d-block<?php echo isset($errors['tauliah_category']) ? '' : ' d-none'; ?>" data-error-for="tauliah_category">
                                            <?php echo isset($errors['tauliah_category']) ? htmlspecialchars($errors['tauliah_category'], ENT_QUOTES, 'UTF-8') : ''; ?>
                                        </div>
                                    </div>
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label for="state_id" class="form-label">Negeri Mengajar</label>
                                            <select class="form-select<?php echo isset($errors['state_id']) ? ' is-invalid' : ''; ?>"
                                                    id="state_id" name="state_id" required>
                                                <?php foreach ($states as $id => $name): ?>
                                                    <option value="<?php echo $id; ?>"
                                                        <?php echo (int) $formData['state_id'] === $id ? 'selected' : ''; ?>>
                                                        <?php echo htmlspecialchars($name, ENT_QUOTES, 'UTF-8'); ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                            <div class="invalid-feedback d-block<?php echo isset($errors['state_id']) ? '' : ' d-none'; ?>" data-error-for="state_id">
                                                <?php echo isset($errors['state_id']) ? htmlspecialchars($errors['state_id'], ENT_QUOTES, 'UTF-8') : ''; ?>
                                            </div>
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label for="district_id" class="form-label">Daerah/Kawasan</label>
                                            <select class="form-select<?php echo isset($errors['district_id']) ? ' is-invalid' : ''; ?>"
                                                    id="district_id" name="district_id">
                                                <option value=""<?php echo $formData['district_id'] === '' ? ' selected' : ''; ?>>Seluruh Negeri Melaka</option>
                                            </select>
                                            <small class="form-text text-muted<?php echo isset($errors['district_id']) ? ' d-none' : ''; ?>" data-helper-for="district_id">Daerah akan dipaparkan mengikut negeri yang dipilih.</small>
                                            <div class="invalid-feedback d-block<?php echo isset($errors['district_id']) ? '' : ' d-none'; ?>" data-error-for="district_id">
                                                <?php echo isset($errors['district_id']) ? htmlspecialchars($errors['district_id'], ENT_QUOTES, 'UTF-8') : ''; ?>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="mb-3">
                                        <label for="teaching_location_detail" class="form-label">Lokasi Pengajaran (Butiran)</label>
                                        <textarea id="teaching_location_detail" name="teaching_location_detail" rows="2" class="form-control"><?php echo htmlspecialchars($formData['teaching_location_detail'], ENT_QUOTES, 'UTF-8'); ?></textarea>
                                        <small class="form-text text-muted">Contoh: Masjid Ar-Rahman, Sekolah Rendah Islam Melaka.</small>
                                    </div>
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label for="experience_years" class="form-label">Tempoh Pengalaman Mengajar (Tahun)</label>
                                            <input type="number" min="0" max="60" step="1"
                                                   class="form-control<?php echo isset($errors['experience_years']) ? ' is-invalid' : ''; ?>"
                                                   id="experience_years" name="experience_years"
                                                   value="<?php echo htmlspecialchars($formData['experience_years'], ENT_QUOTES, 'UTF-8'); ?>">
                                            <div class="invalid-feedback d-block<?php echo isset($errors['experience_years']) ? '' : ' d-none'; ?>" data-error-for="experience_years">
                                                <?php echo isset($errors['experience_years']) ? htmlspecialchars($errors['experience_years'], ENT_QUOTES, 'UTF-8') : ''; ?>
                                            </div>
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label for="subjects" class="form-label">Bidang/Dapatan Pengajaran</label>
                                            <input type="text" id="subjects" name="subjects" class="form-control"
                                                   value="<?php echo htmlspecialchars($formData['subjects'], ENT_QUOTES, 'UTF-8'); ?>">
                                            <small class="form-text text-muted">Contoh: Fiqh, Tajwid, Hafazan.</small>
                                        </div>
                                    </div>
                                    <div class="mb-3">
                                        <label for="notes" class="form-label">Catatan Tambahan</label>
                                        <textarea id="notes" name="notes" rows="3" class="form-control"><?php echo htmlspecialchars($formData['notes'], ENT_QUOTES, 'UTF-8'); ?></textarea>
                                    </div>
                                    <div class="alert alert-info">
                                        <i class="ti-info-alt"></i> Dengan menghantar permohonan ini, anda bersetuju bahawa segala maklumat yang diberikan adalah benar. Pentadbir akan memaklumkan perkembangan melalui portal ini dan e-mel berdaftar.
                                    </div>
                                    <div class="text-end">
                                        <button type="submit" class="btn btn-danger" data-role="submit-application">
                                            <i class="ti-check"></i> Hantar Permohonan
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-4 col-md-5 col-12">
                        <div class="card">
                            <div class="card-body">
                                <h4 class="card-title">Syarat &amp; Garis Panduan</h4>
                                <p class="text-muted mb-3">Pastikan maklumat profil anda lengkap terutamanya nombor kad pengenalan, alamat surat-menyurat dan kelayakan akademik. Semua dokumen sokongan hendaklah disediakan untuk semakan seterusnya.</p>
                                <ul class="list-unstyled small text-muted">
                                    <li><i class="ti-check text-success me-1"></i> Semak semula pilihan jenis dan kategori permohonan.</li>
                                    <li><i class="ti-check text-success me-1"></i> Pastikan lokasi pengajaran dinyatakan dengan jelas.</li>
                                    <li><i class="ti-check text-success me-1"></i> Permohonan pembaharuan hanya dibenarkan untuk tauliah aktif.</li>
                                </ul>
                            </div>
                        </div>
                        <div class="card">
                            <div class="card-body">
                                <h4 class="card-title">Pengumuman Terkini</h4>
                                <?php if (!$announcements): ?>
                                    <p class="text-muted mb-0">Tiada pengumuman terkini untuk pemohon.</p>
                                <?php else: ?>
                                    <div class="list-group">
                                        <?php foreach ($announcements as $announcement): ?>
                                            <div class="list-group-item">
                                                <h6 class="mb-1"><?php echo htmlspecialchars($announcement['title'], ENT_QUOTES, 'UTF-8'); ?></h6>
                                                <small class="text-muted d-block mb-1">
                                                    <?php echo htmlspecialchars(date('d/m/Y', strtotime((string) $announcement['published_at'])), ENT_QUOTES, 'UTF-8'); ?>
                                                </small>
                                                <div class="announcement-body small text-muted mb-0">
                                                    <?php echo (string) ($announcement['body'] ?? ''); ?>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
<?php
ob_start();
?>
<script>
    (function () {
        var districtsByState = <?php echo $districtsJson ?: '{}'; ?>;
        var defaultStateId = <?php echo json_encode($defaultStateId, JSON_UNESCAPED_UNICODE); ?>;
        var stateSelect = document.getElementById('state_id');
        var districtSelect = document.getElementById('district_id');

        function renderDistrictOptions(stateId) {
            if (!districtSelect) {
                return;
            }

            var selectedValue = <?php echo json_encode((string) $formData['district_id'], JSON_UNESCAPED_UNICODE); ?>;
            districtSelect.innerHTML = '';

            var defaultOption = document.createElement('option');
            defaultOption.value = '';
            defaultOption.textContent = 'Seluruh Negeri Melaka';
            if (!selectedValue) {
                defaultOption.selected = true;
            }
            districtSelect.appendChild(defaultOption);

            var districts = districtsByState[stateId] || [];
            districts.forEach(function (district) {
                var option = document.createElement('option');
                option.value = district.id;
                option.textContent = district.name;
                if (String(district.id) === String(selectedValue)) {
                    option.selected = true;
                }
                districtSelect.appendChild(option);
            });
        }

        if (stateSelect && !stateSelect.value && defaultStateId) {
            stateSelect.value = defaultStateId;
        }

        var initialState = stateSelect ? (stateSelect.value || defaultStateId) : defaultStateId;
        if (initialState) {
            renderDistrictOptions(initialState);
        }

        if (stateSelect) {
            stateSelect.addEventListener('change', function (event) {
                renderDistrictOptions(event.target.value || '');
            });
        }

        var nricField = document.getElementById('nric');
        if (nricField) {
            nricField.addEventListener('input', function () {
                this.value = this.value.replace(/[^0-9]/g, '').slice(0, 12);
            });
        }

        var form = document.querySelector('[data-role="application-form"]');
        if (form) {
            var submitBtn = form.querySelector('[data-role="submit-application"]');
            var alertBox = document.getElementById('application-alert');

            var toggleAlert = function (message, type) {
                if (!alertBox) {
                    return;
                }
                alertBox.classList.remove('alert-danger', 'alert-success', 'alert-info');
                if (!message) {
                    alertBox.classList.add('d-none');
                    alertBox.textContent = '';
                    return;
                }
                var alertType = 'danger';
                if (type === 'success') {
                    alertType = 'success';
                } else if (type === 'info') {
                    alertType = 'info';
                }
                alertBox.classList.remove('d-none');
                alertBox.classList.add('alert-' + alertType);
                alertBox.textContent = message;
            };

            var clearFieldErrors = function () {
                var invalidElements = form.querySelectorAll('.is-invalid');
                invalidElements.forEach(function (element) {
                    element.classList.remove('is-invalid');
                });

                var feedbacks = form.querySelectorAll('[data-error-for]');
                feedbacks.forEach(function (element) {
                    element.classList.add('d-none');
                    element.textContent = '';
                });

                var helpers = form.querySelectorAll('[data-helper-for]');
                helpers.forEach(function (element) {
                    element.classList.remove('d-none');
                });
            };

            var applyFieldErrors = function (errors) {
                if (!errors) {
                    return;
                }

                Object.keys(errors).forEach(function (key) {
                    if (key === '_global') {
                        toggleAlert(errors[key], 'danger');
                        return;
                    }

                    var field = form.querySelector('[name="' + key + '"]');
                    if (field) {
                        field.classList.add('is-invalid');
                    }

                    var feedback = form.querySelector('[data-error-for="' + key + '"]');
                    if (feedback) {
                        feedback.textContent = errors[key];
                        feedback.classList.remove('d-none');
                    }

                    var helper = form.querySelector('[data-helper-for="' + key + '"]');
                    if (helper) {
                        helper.classList.add('d-none');
                    }
                });
            };

            form.addEventListener('submit', function (event) {
                event.preventDefault();
                toggleAlert('', '');
                clearFieldErrors();

                if (submitBtn) {
                    submitBtn.disabled = true;
                    submitBtn.classList.add('disabled');
                }

                var formData = new FormData(form);

                fetch(form.action || window.location.href, {
                    method: 'POST',
                    body: formData,
                    credentials: 'same-origin',
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest',
                        'Accept': 'application/json'
                    }
                })
                    .then(function (response) {
                        return response
                            .json()
                            .catch(function () {
                                return {};
                            })
                            .then(function (data) {
                                return {
                                    ok: response.ok,
                                    status: response.status,
                                    data: data
                                };
                            });
                    })
                    .then(function (result) {
                        var data = result.data || {};

                        if (result.ok && data.success) {
                            toggleAlert(data.message || 'Permohonan berjaya dihantar.', 'success');
                            var redirectUrl = data.redirect || '/pemohon/applications/index.php';
                            setTimeout(function () {
                                window.location.href = redirectUrl;
                            }, 600);
                            return;
                        }

                        var message = data.message || 'Permohonan tidak dapat dihantar buat masa ini. Sila cuba lagi atau hubungi pentadbir.';
                        toggleAlert(message, 'danger');
                        applyFieldErrors(data.errors || {});
                    })
                    .catch(function (error) {
                        console.error('Permohonan AJAX gagal:', error);
                        toggleAlert('Permohonan tidak dapat dihantar buat masa ini. Sila cuba lagi atau hubungi pentadbir.', 'danger');
                    })
                    .finally(function () {
                        if (submitBtn) {
                            submitBtn.disabled = false;
                            submitBtn.classList.remove('disabled');
                        }
                    });
            });
        }
    }());
</script>
<?php
$pageInlineScripts = ob_get_clean();

            include __DIR__ . '/../footer.php';
