<?php

declare(strict_types=1);

require_once __DIR__ . '/includes/helpers.php';
require_once __DIR__ . '/includes/auth.php';
require_once __DIR__ . '/includes/education.php';

pemohon_require_login();

$currentUser = pemohon_current_user();
if ($currentUser === null) {
    http_response_code(401);
    exit;
}

$userId = (int) ($currentUser['id'] ?? 0);
if ($userId <= 0) {
    http_response_code(401);
    exit;
}

$method = strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');

if ($method === 'GET') {
    header('Content-Type: application/json; charset=utf-8');

    try {
        $items = pemohon_fetch_educations($userId);
    } catch (Throwable $exception) {
        http_response_code(500);
        echo json_encode([
            'message' => 'Gagal memuatkan senarai kelayakan akademik.',
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    echo json_encode([
        'items' => array_map('pemohon_format_education_item', $items),
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

header('Content-Type: application/json; charset=utf-8');

$token = $_POST['_token'] ?? null;
if (!pemohon_validate_csrf_token(is_string($token) ? $token : null)) {
    http_response_code(419);
    echo json_encode([
        'message' => 'Sesi tidak sah. Sila muat semula halaman.',
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$action = strtolower(trim((string) ($_POST['_action'] ?? $_POST['action'] ?? '')));

if ($action === 'reorder') {
    $orderPayload = $_POST['order'] ?? '[]';
    if (is_string($orderPayload)) {
        $decoded = json_decode($orderPayload, true);
    } else {
        $decoded = $orderPayload;
    }

    if (!is_array($decoded)) {
        http_response_code(422);
        echo json_encode([
            'message' => 'Susunan baharu tidak sah.',
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    try {
        $success = pemohon_reorder_educations($userId, $decoded);
    } catch (Throwable $exception) {
        http_response_code(500);
        echo json_encode([
            'message' => 'Gagal mengemaskini susunan kelayakan.',
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    if (!$success) {
        http_response_code(400);
        echo json_encode([
            'message' => 'Susunan kelayakan tidak dikemaskini.',
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    try {
        $items = pemohon_fetch_educations($userId);
    } catch (Throwable $exception) {
        $items = [];
    }

    echo json_encode([
        'message' => 'Susunan kelayakan berjaya dikemaskini.',
        'items' => array_map('pemohon_format_education_item', $items),
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

if ($action === 'delete') {
    $educationId = (int) ($_POST['id'] ?? 0);
    if ($educationId <= 0) {
        http_response_code(422);
        echo json_encode([
            'message' => 'Rekod kelayakan tidak sah.',
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    try {
        $deleted = pemohon_delete_education($userId, $educationId);
    } catch (Throwable $exception) {
        http_response_code(500);
        echo json_encode([
            'message' => 'Gagal menghapuskan rekod kelayakan.',
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    if (!$deleted) {
        http_response_code(404);
        echo json_encode([
            'message' => 'Rekod kelayakan tidak ditemui.',
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    echo json_encode([
        'message' => 'Rekod kelayakan berjaya dipadam.',
        'id' => $educationId,
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$isUpdate = $action === 'update';
$educationId = $isUpdate ? (int) ($_POST['id'] ?? 0) : 0;

if ($isUpdate && $educationId <= 0) {
    http_response_code(422);
    echo json_encode([
        'message' => 'Rekod kelayakan tidak sah.',
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$institution = trim((string) ($_POST['institution'] ?? ''));
$qualification = trim((string) ($_POST['qualification'] ?? ''));
$fieldOfStudy = trim((string) ($_POST['field_of_study'] ?? ''));
$startYear = trim((string) ($_POST['start_year'] ?? ''));
$endYear = trim((string) ($_POST['end_year'] ?? ''));
$description = trim((string) ($_POST['description'] ?? ''));
$isHighest = isset($_POST['is_highest']) && ($_POST['is_highest'] === 'on' || $_POST['is_highest'] === '1' || $_POST['is_highest'] === 'true');

$errors = [];

if ($institution === '') {
    $errors['institution'] = 'Nama institusi diperlukan.';
} elseif (mb_strlen($institution) > 200) {
    $errors['institution'] = 'Nama institusi melebihi 200 aksara.';
}

if ($qualification === '') {
    $errors['qualification'] = 'Nama kelayakan diperlukan.';
} elseif (mb_strlen($qualification) > 150) {
    $errors['qualification'] = 'Nama kelayakan melebihi 150 aksara.';
}

if ($fieldOfStudy !== '' && mb_strlen($fieldOfStudy) > 150) {
    $errors['field_of_study'] = 'Bidang pengajian melebihi 150 aksara.';
}

$startYearValue = null;
if ($startYear !== '') {
    if (!preg_match('/^\d{4}$/', $startYear)) {
        $errors['start_year'] = 'Tahun mula tidak sah.';
    } else {
        $startYearValue = (int) $startYear;
    }
}

$endYearValue = null;
if ($endYear !== '') {
    if (!preg_match('/^\d{4}$/', $endYear)) {
        $errors['end_year'] = 'Tahun tamat tidak sah.';
    } else {
        $endYearValue = (int) $endYear;
    }
}

if ($startYearValue !== null && $endYearValue !== null && $startYearValue > $endYearValue) {
    $errors['end_year'] = 'Tahun tamat mestilah selepas atau sama dengan tahun mula.';
}

if ($description !== '' && mb_strlen($description) > 500) {
    $errors['description'] = 'Penerangan melebihi 500 aksara.';
}

if ($errors) {
    http_response_code(422);
    echo json_encode([
        'message' => 'Sila betulkan maklumat yang diberikan.',
        'errors' => $errors,
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$payload = [
    'institution' => $institution,
    'qualification' => $qualification,
    'field_of_study' => $fieldOfStudy !== '' ? $fieldOfStudy : null,
    'start_year' => $startYearValue,
    'end_year' => $endYearValue,
    'description' => $description !== '' ? $description : null,
    'is_highest' => $isHighest,
];

try {
    if ($isUpdate) {
        $record = pemohon_update_education($userId, $educationId, $payload);
    } else {
        $record = pemohon_insert_education($userId, $payload);
    }
} catch (Throwable $exception) {
    http_response_code(500);
    echo json_encode([
        'message' => 'Gagal menyimpan rekod kelayakan.',
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

if ($record === null) {
    http_response_code(400);
    echo json_encode([
        'message' => 'Maklumat pemohon tidak ditemui.',
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

echo json_encode([
    'message' => $isUpdate ? 'Rekod kelayakan berjaya dikemaskini.' : 'Rekod kelayakan berjaya ditambah.',
    'item' => pemohon_format_education_item($record),
], JSON_UNESCAPED_UNICODE);
exit;

/**
 * @param array<string, mixed>|null $row
 */
function pemohon_format_education_item(?array $row): ?array
{
    if (!$row) {
        return null;
    }

    return [
        'id' => (int) ($row['id'] ?? 0),
        'institution' => (string) ($row['institution'] ?? ''),
        'qualification' => (string) ($row['qualification'] ?? ''),
        'field_of_study' => $row['field_of_study'] ?? null,
        'start_year' => isset($row['start_year']) ? (int) $row['start_year'] : null,
        'end_year' => isset($row['end_year']) ? (int) $row['end_year'] : null,
        'description' => $row['description'] ?? null,
        'display_order' => isset($row['display_order']) ? (int) $row['display_order'] : 0,
        'is_highest' => isset($row['is_highest']) ? (bool) $row['is_highest'] : false,
        'created_at' => $row['created_at'] ?? null,
        'updated_at' => $row['updated_at'] ?? null,
    ];
}
