<?php

declare(strict_types=1);

require_once __DIR__ . '/db.php';

function pemohon_login(string $email, string $password): bool
{
    $email = strtolower(trim($email));
    if ($email === '' || $password === '') {
        return false;
    }

    $db = pemohon_db();
    $sql = 'SELECT u.id, u.uuid, u.name, u.email, u.password, u.status,
                   u.email_verified_at, u.nric, u.phone,
                   a.photo_path
            FROM users u
            INNER JOIN role_user ru ON ru.user_id = u.id AND ru.is_primary = 1
            INNER JOIN roles r ON r.id = ru.role_id
            LEFT JOIN applicants a ON a.user_id = u.id
            WHERE u.email = ? AND r.name = ?
            LIMIT 1';

    $stmt = $db->prepare($sql);
    if (!$stmt) {
        throw new \RuntimeException('Failed to prepare statement: ' . $db->error);
    }

    $role = 'applicant';
    $stmt->bind_param('ss', $email, $role);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $stmt->close();

    if (!$user) {
        return false;
    }

    if (!password_verify($password, $user['password'] ?? '')) {
        return false;
    }

    if (($user['status'] ?? 'pending') !== 'active') {
        return false;
    }

    pemohon_start_session();
    $_SESSION['pemohon_user'] = [
        'id' => (int) $user['id'],
        'uuid' => $user['uuid'],
        'name' => $user['name'],
        'email' => $user['email'],
        'status' => $user['status'],
        'email_verified_at' => $user['email_verified_at'],
        'nric' => $user['nric'] ?? null,
        'phone' => $user['phone'] ?? null,
        'photo_path' => $user['photo_path'] ?? null,
    ];

    return true;
}

function pemohon_logout(): void
{
    pemohon_start_session();
    unset($_SESSION['pemohon_user']);
}

function pemohon_current_user(): ?array
{
    pemohon_start_session();
    return $_SESSION['pemohon_user'] ?? null;
}

function pemohon_require_login(): void
{
    if (!pemohon_current_user()) {
        pemohon_redirect('/pemohon');
    }
}

function pemohon_fetch_profile(int $userId): ?array
{
    $db = pemohon_db();
    $sql = 'SELECT a.*, u.email, u.name, u.phone AS user_phone
            FROM applicants a
            INNER JOIN users u ON u.id = a.user_id
            WHERE u.id = ?
            LIMIT 1';

    $stmt = $db->prepare($sql);
    if (!$stmt) {
        throw new \RuntimeException('Failed to prepare statement: ' . $db->error);
    }

    $stmt->bind_param('i', $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    $profile = $result->fetch_assoc() ?: null;
    $stmt->close();

    if ($profile !== null) {
        pemohon_start_session();
        if (!isset($_SESSION['pemohon_user'])) {
            $_SESSION['pemohon_user'] = [
                'id' => (int) $profile['user_id'],
            ];
        }
        $_SESSION['pemohon_user']['photo_path'] = $profile['photo_path'] ?? ($_SESSION['pemohon_user']['photo_path'] ?? null);
        $_SESSION['pemohon_user']['nric'] = $profile['nric'] ?? ($_SESSION['pemohon_user']['nric'] ?? null);
        $_SESSION['pemohon_user']['phone'] = $profile['phone'] ?? $profile['user_phone'] ?? ($_SESSION['pemohon_user']['phone'] ?? null);
    }

    return $profile;
}
