<?php

declare(strict_types=1);

require_once __DIR__ . '/db.php';
require_once __DIR__ . '/applicant.php';

/**
 * Fetches document rows for the current applicant.
 *
 * @return array<int, array<string, mixed>>
 */
function pemohon_fetch_documents(int $userId): array
{
    $db = pemohon_db();

    $sql = 'SELECT d.id, d.title, d.description, d.original_filename, d.mime_type,
                   d.file_size, d.stored_path, d.created_at, d.updated_at
            FROM applicant_documents d
            INNER JOIN applicants a ON a.id = d.applicant_id
            WHERE a.user_id = ?
            ORDER BY d.created_at DESC, d.id DESC';

    $stmt = $db->prepare($sql);
    if (!$stmt) {
        if ((int) $db->errno === 1146) {
            return [];
        }
        throw new \RuntimeException('Failed to prepare statement: ' . $db->error);
    }

    $stmt->bind_param('i', $userId);
    $stmt->execute();
    $result = $stmt->get_result();

    $documents = [];
    while ($row = $result->fetch_assoc()) {
        $documents[] = $row;
    }

    $stmt->close();

    return $documents;
}

/**
 * Retrieves a document record ensuring it belongs to the given user.
 */
function pemohon_find_document(int $userId, int $documentId): ?array
{
    if ($documentId <= 0) {
        return null;
    }

    $db = pemohon_db();

    $sql = 'SELECT d.id, d.title, d.description, d.original_filename, d.mime_type,
                   d.file_size, d.stored_path, d.created_at, d.updated_at, d.applicant_id
            FROM applicant_documents d
            INNER JOIN applicants a ON a.id = d.applicant_id
            WHERE d.id = ? AND a.user_id = ?
            LIMIT 1';

    $stmt = $db->prepare($sql);
    if (!$stmt) {
        if ((int) $db->errno === 1146) {
            return null;
        }
        throw new \RuntimeException('Failed to prepare statement: ' . $db->error);
    }

    $stmt->bind_param('ii', $documentId, $userId);
    $stmt->execute();
    $result = $stmt->get_result();

    $document = $result->fetch_assoc() ?: null;
    $stmt->close();

    return $document;
}
