<?php

declare(strict_types=1);

require_once __DIR__ . '/includes/helpers.php';
require_once __DIR__ . '/includes/auth.php';

pemohon_require_login();

$currentUser = pemohon_current_user();
if ($currentUser === null) {
    pemohon_redirect('/pemohon');
}

$db = pemohon_db();
$userId = (int) $currentUser['id'];
$profile = pemohon_fetch_profile($userId) ?? [];

$userNric = '';
$userNricStmt = $db->prepare('SELECT nric FROM users WHERE id = ? LIMIT 1');
if ($userNricStmt) {
    $userNricStmt->bind_param('i', $userId);
    $userNricStmt->execute();
    $userNricStmt->bind_result($userNricValue);
    if ($userNricStmt->fetch()) {
        $userNric = (string) $userNricValue;
    }
    $userNricStmt->close();
}

$isAjax = isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strcasecmp((string) $_SERVER['HTTP_X_REQUESTED_WITH'], 'xmlhttprequest') === 0;
$isModal = $isAjax || (isset($_GET['modal']) && $_GET['modal'] === '1');

$formData = [
    'name' => $currentUser['name'] ?? '',
    'email' => $currentUser['email'] ?? '',
    'nric' => $profile['nric'] ?? $userNric,
    'phone' => $profile['phone'] ?? '',
    'date_of_birth' => $profile['date_of_birth'] ?? '',
    'occupation' => $profile['occupation'] ?? '',
    'address_line1' => $profile['address_line1'] ?? '',
    'address_line2' => $profile['address_line2'] ?? '',
    'city' => $profile['city'] ?? '',
    'state' => $profile['state'] ?? '',
    'postcode' => $profile['postcode'] ?? '',
    'country' => $profile['country'] ?? '',
    'biography' => $profile['biography'] ?? '',
];

$formData['nric'] = preg_replace('/\D+/', '', (string) ($formData['nric'] ?? ''));

$errors = [];
$message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $formData['name'] = trim((string) ($_POST['name'] ?? ''));
    $formData['nric'] = preg_replace('/\D+/', '', (string) ($_POST['nric'] ?? ''));
    $formData['phone'] = trim((string) ($_POST['phone'] ?? ''));
    $formData['date_of_birth'] = trim((string) ($_POST['date_of_birth'] ?? ''));
    $formData['occupation'] = trim((string) ($_POST['occupation'] ?? ''));
    $formData['address_line1'] = trim((string) ($_POST['address_line1'] ?? ''));
    $formData['address_line2'] = trim((string) ($_POST['address_line2'] ?? ''));
    $formData['city'] = trim((string) ($_POST['city'] ?? ''));
    $formData['state'] = trim((string) ($_POST['state'] ?? ''));
    $formData['postcode'] = trim((string) ($_POST['postcode'] ?? ''));
    $formData['country'] = trim((string) ($_POST['country'] ?? ''));
    $formData['biography'] = trim((string) ($_POST['biography'] ?? ''));

    if ($formData['name'] === '') {
        $errors['name'] = 'Nama penuh diperlukan.';
    }

    if ($formData['nric'] === '') {
        $errors['nric'] = 'No. Kad Pengenalan diperlukan.';
    } elseif (!preg_match('/^\d{12}$/', $formData['nric'])) {
        $errors['nric'] = 'No. Kad Pengenalan hendaklah terdiri daripada 12 digit.';
    }

    if ($formData['phone'] !== '' && !preg_match('/^[0-9\\-\\+\\s]{6,20}$/', $formData['phone'])) {
        $errors['phone'] = 'Sila masukkan nombor telefon yang sah.';
    }

    if ($formData['date_of_birth'] !== '' && !preg_match('/^\\d{4}-\\d{2}-\\d{2}$/', $formData['date_of_birth'])) {
        $errors['date_of_birth'] = 'Tarikh lahir hendaklah dalam format YYYY-MM-DD.';
    }

    if (!$errors && $formData['nric'] !== '') {
        $uniqueStmt = $db->prepare('SELECT id FROM users WHERE nric = ? AND id <> ? LIMIT 1');
        if (!$uniqueStmt) {
            $errors['general'] = 'Gagal mengesahkan No. Kad Pengenalan. Sila cuba lagi.';
        } else {
            $uniqueStmt->bind_param('si', $formData['nric'], $userId);
            $uniqueStmt->execute();
            $uniqueStmt->store_result();
            if ($uniqueStmt->num_rows > 0) {
                $errors['nric'] = 'No. Kad Pengenalan telah digunakan oleh akaun lain.';
            }
            $uniqueStmt->close();
        }
    }

    if (!$errors) {
        $db->begin_transaction();
        try {
            $userStmt = $db->prepare('UPDATE users SET name = ?, phone = NULLIF(?, \'\'), nric = NULLIF(?, \'\') WHERE id = ?');
            if (!$userStmt) {
                throw new \RuntimeException('Gagal menyediakan kemaskini pengguna: ' . $db->error);
            }
            $userStmt->bind_param('sssi', $formData['name'], $formData['phone'], $formData['nric'], $userId);
            $userStmt->execute();
            $userStmt->close();

            $applicantExists = !empty($profile);
            if ($applicantExists) {
                $appStmt = $db->prepare(
                    'UPDATE applicants
                     SET nric = NULLIF(?, \'\'),
                         date_of_birth = NULLIF(?, \'\'),
                         occupation = NULLIF(?, \'\'),
                         address_line1 = NULLIF(?, \'\'),
                         address_line2 = NULLIF(?, \'\'),
                         city = NULLIF(?, \'\'),
                         state = NULLIF(?, \'\'),
                         postcode = NULLIF(?, \'\'),
                         country = NULLIF(?, \'\'),
                         biography = NULLIF(?, \'\')
                     WHERE user_id = ?'
                );

                if (!$appStmt) {
                    throw new \RuntimeException('Gagal menyediakan kemaskini pemohon: ' . $db->error);
                }

                $appStmt->bind_param(
                    'ssssssssssi',
                    $formData['nric'],
                    $formData['date_of_birth'],
                    $formData['occupation'],
                    $formData['address_line1'],
                    $formData['address_line2'],
                    $formData['city'],
                    $formData['state'],
                    $formData['postcode'],
                    $formData['country'],
                    $formData['biography'],
                    $userId
                );
                $appStmt->execute();
                $appStmt->close();
            } else {
                $appStmt = $db->prepare(
                    'INSERT INTO applicants (
                        user_id,
                        nric,
                        date_of_birth,
                        occupation,
                        address_line1,
                        address_line2,
                        city,
                        state,
                        postcode,
                        country,
                        biography
                    ) VALUES (
                        ?,
                        NULLIF(?, \'\'),
                        NULLIF(?, \'\'),
                        NULLIF(?, \'\'),
                        NULLIF(?, \'\'),
                        NULLIF(?, \'\'),
                        NULLIF(?, \'\'),
                        NULLIF(?, \'\'),
                        NULLIF(?, \'\'),
                        NULLIF(?, \'\'),
                        NULLIF(?, \'\')
                    )'
                );

                if (!$appStmt) {
                    throw new \RuntimeException('Gagal menyediakan tambah pemohon: ' . $db->error);
                }

                $appStmt->bind_param(
                    'issssssssss',
                    $userId,
                    $formData['nric'],
                    $formData['date_of_birth'],
                    $formData['occupation'],
                    $formData['address_line1'],
                    $formData['address_line2'],
                    $formData['city'],
                    $formData['state'],
                    $formData['postcode'],
                    $formData['country'],
                    $formData['biography']
                );
                $appStmt->execute();
                $appStmt->close();
            }

            $db->commit();

            $_SESSION['pemohon_user']['name'] = $formData['name'];
            if ($formData['phone'] !== '') {
                $_SESSION['pemohon_user']['phone'] = $formData['phone'];
            } else {
                unset($_SESSION['pemohon_user']['phone']);
            }
            $_SESSION['pemohon_user']['nric'] = $formData['nric'];

            $profile = pemohon_fetch_profile($userId) ?? [];
            if (!empty($profile['photo_path'])) {
                $_SESSION['pemohon_user']['photo_path'] = $profile['photo_path'];
            }
            $message = 'Profil berjaya dikemaskini.';

            if ($isAjax) {
                header('Content-Type: application/json; charset=utf-8');
                echo json_encode([
                    'success' => true,
                    'message' => $message,
                ]);
                exit;
            }

            $_SESSION['pemohon_message'] = $message;
            pemohon_redirect('/pemohon/profile.php');
        } catch (\Throwable $throwable) {
            $db->rollback();
            $errors['general'] = 'Gagal mengemaskini profil. Sila cuba lagi. (' . $throwable->getMessage() . ')';

            if ($isAjax) {
                header('Content-Type: application/json; charset=utf-8', true, 500);
                echo json_encode([
                    'success' => false,
                    'message' => $errors['general'],
                ]);
                exit;
            }
        }
    } elseif ($isAjax) {
        header('Content-Type: application/json; charset=utf-8', true, 422);
        echo json_encode([
            'success' => false,
            'message' => 'Sila semak ruangan yang ditandakan.',
            'html' => render_update_profile_form($formData, $errors, '', true),
        ]);
        exit;
    }
}

$formHtml = render_update_profile_form($formData, $errors, $message, $isModal);

if ($isModal) {
    echo $formHtml;
    exit;
}

$pageTitle = 'Kemaskini Profil Pemohon';
include __DIR__ . '/header.php';
include __DIR__ . '/sidebar.php';
?>
        <div class="page-wrapper">
            <div class="container-fluid">
                <div class="row">
                    <div class="col-lg-8 col-md-10 mx-auto">
                        <div class="card">
                            <div class="card-body">
                                <h4 class="card-title">Kemaskini Profil</h4>
                                <p class="card-subtitle">Sila pastikan maklumat yang dimasukkan adalah tepat.</p>
                                <?php echo $formHtml; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
<?php
include __DIR__ . '/footer.php';
exit;

function render_update_profile_form(array $data, array $errors = [], string $message = '', bool $isModal = false): string
{
    $errorAlert = '';
    if (!empty($errors['general'])) {
        $errorAlert = '<div class="alert alert-danger">' . htmlspecialchars($errors['general'], ENT_QUOTES, 'UTF-8') . '</div>';
    }

    if ($message !== '') {
        $errorAlert = '<div class="alert alert-success">' . htmlspecialchars($message, ENT_QUOTES, 'UTF-8') . '</div>';
    }

    $getValue = static function (string $key) use ($data): string {
        return htmlspecialchars($data[$key] ?? '', ENT_QUOTES, 'UTF-8');
    };

    $fieldError = static function (string $key) use ($errors): string {
        if (!isset($errors[$key])) {
            return '';
        }

        return '<div class="invalid-feedback d-block">' . htmlspecialchars($errors[$key], ENT_QUOTES, 'UTF-8') . '</div>';
    };

    ob_start();
    ?>
    <form id="updateProfileForm" method="post" action="/pemohon/update-profile.php" novalidate>
        <?php echo $errorAlert; ?>
        <div class="row g-3">
            <div class="col-12">
                <label for="profile-name" class="form-label">Nama Penuh <span class="text-danger">*</span></label>
                <input type="text" id="profile-name" name="name" class="form-control<?php echo isset($errors['name']) ? ' is-invalid' : ''; ?>" required value="<?php echo $getValue('name'); ?>">
                <?php echo $fieldError('name'); ?>
            </div>
            <div class="col-md-6">
                <label for="profile-nric" class="form-label">No. Kad Pengenalan <span class="text-danger">*</span></label>
                <input type="text" id="profile-nric" name="nric" maxlength="12" inputmode="numeric" pattern="\d{12}" class="form-control<?php echo isset($errors['nric']) ? ' is-invalid' : ''; ?>" value="<?php echo $getValue('nric'); ?>" placeholder="Contoh: 900101015555" required>
                <small class="form-text text-muted">Masukkan 12 digit tanpa sengkang.</small>
                <?php echo $fieldError('nric'); ?>
            </div>
            <div class="col-md-6">
                <label for="profile-email" class="form-label">Emel</label>
                <input type="email" id="profile-email" name="email" class="form-control" value="<?php echo $getValue('email'); ?>" readonly>
            </div>
            <div class="col-md-6">
                <label for="profile-phone" class="form-label">Telefon</label>
                <input type="text" id="profile-phone" name="phone" class="form-control<?php echo isset($errors['phone']) ? ' is-invalid' : ''; ?>" value="<?php echo $getValue('phone'); ?>" placeholder="+6012 345 6789">
                <?php echo $fieldError('phone'); ?>
            </div>
            <div class="col-md-6">
                <label for="profile-dob" class="form-label">Tarikh Lahir</label>
                <input type="date" id="profile-dob" name="date_of_birth" class="form-control<?php echo isset($errors['date_of_birth']) ? ' is-invalid' : ''; ?>" value="<?php echo $getValue('date_of_birth'); ?>">
                <?php echo $fieldError('date_of_birth'); ?>
            </div>
            <div class="col-md-6">
                <label for="profile-occupation" class="form-label">Pekerjaan</label>
                <input type="text" id="profile-occupation" name="occupation" class="form-control" value="<?php echo $getValue('occupation'); ?>">
            </div>
            <div class="col-12">
                <label for="profile-address-line1" class="form-label">Alamat Baris 1</label>
                <input type="text" id="profile-address-line1" name="address_line1" class="form-control" value="<?php echo $getValue('address_line1'); ?>">
            </div>
            <div class="col-12">
                <label for="profile-address-line2" class="form-label">Alamat Baris 2</label>
                <input type="text" id="profile-address-line2" name="address_line2" class="form-control" value="<?php echo $getValue('address_line2'); ?>">
            </div>
            <div class="col-md-4">
                <label for="profile-city" class="form-label">Bandar</label>
                <input type="text" id="profile-city" name="city" class="form-control" value="<?php echo $getValue('city'); ?>">
            </div>
            <div class="col-md-4">
                <label for="profile-state" class="form-label">Negeri</label>
                <input type="text" id="profile-state" name="state" class="form-control" value="<?php echo $getValue('state'); ?>">
            </div>
            <div class="col-md-4">
                <label for="profile-postcode" class="form-label">Poskod</label>
                <input type="text" id="profile-postcode" name="postcode" class="form-control" value="<?php echo $getValue('postcode'); ?>">
            </div>
            <div class="col-md-6">
                <label for="profile-country" class="form-label">Negara</label>
                <input type="text" id="profile-country" name="country" class="form-control" value="<?php echo $getValue('country'); ?>">
            </div>
            <div class="col-12">
                <label for="profile-biography" class="form-label">Biografi</label>
                <textarea id="profile-biography" name="biography" rows="4" class="form-control"><?php echo $getValue('biography'); ?></textarea>
            </div>
        </div>
        <?php if (!$isModal): ?>
            <div class="mt-4 d-flex justify-content-end gap-2">
                <a href="/pemohon/profile.php" class="btn btn-light border">Kembali</a>
                <button type="submit" class="btn btn-danger">Simpan Perubahan</button>
            </div>
        <?php endif; ?>
    </form>
    <?php if ($isModal): ?>
        <div data-modal-footer="true" class="d-flex justify-content-end gap-2">
            <button type="button" class="btn btn-light border" data-bs-dismiss="modal">Tutup</button>
            <button type="submit" class="btn btn-danger" form="updateProfileForm" data-role="save-profile">Simpan Perubahan</button>
        </div>
    <?php endif; ?>
    <?php

    return (string) ob_get_clean();
}
