<?php
declare(strict_types=1);

require_once __DIR__ . '/includes/helpers.php';
require_once __DIR__ . '/includes/auth.php';
pemohon_require_login();

header('Content-Type: application/json; charset=utf-8');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'message' => 'Kaedah tidak dibenarkan.',
    ]);
    exit;
}

$token = $_POST['_token'] ?? null;
if (!pemohon_validate_csrf_token(is_string($token) ? $token : null)) {
    http_response_code(419);
    echo json_encode([
        'message' => 'Sesi tidak sah. Sila muat semula halaman.',
    ]);
    exit;
}

if (!isset($_FILES['photo']['error']) || $_FILES['photo']['error'] !== UPLOAD_ERR_OK) {
    http_response_code(422);
    echo json_encode([
        'message' => 'Sila pilih foto untuk dimuat naik.',
    ]);
    exit;
}

$file = $_FILES['photo'];

if (!is_uploaded_file($file['tmp_name'])) {
    http_response_code(400);
    echo json_encode([
        'message' => 'Muat naik foto tidak sah.',
    ]);
    exit;
}

$maxFileSize = 5 * 1024 * 1024; // 5MB
if ((int) $file['size'] > $maxFileSize) {
    http_response_code(422);
    echo json_encode([
        'message' => 'Saiz foto melebihi had maksimum 5MB.',
    ]);
    exit;
}

$finfo = new finfo(FILEINFO_MIME_TYPE);
$mimeType = $finfo->file($file['tmp_name']);

$allowedTypes = [
    'image/jpeg' => 'jpg',
    'image/png' => 'png',
];

if (!isset($allowedTypes[$mimeType])) {
    http_response_code(422);
    echo json_encode([
        'message' => 'Hanya foto JPEG atau PNG dibenarkan.',
    ]);
    exit;
}

$imageResource = loadImageResource($file['tmp_name'], $mimeType);
if ($imageResource === null) {
    http_response_code(500);
    echo json_encode([
        'message' => 'Tidak dapat memproses foto yang dimuat naik.',
    ]);
    exit;
}

try {
    $imageResource = correctImageOrientation($imageResource, $file['tmp_name'], $mimeType);
} catch (RuntimeException $exception) {
    // Orientation correction failure should not block the upload; logable if needed.
}

$normalizedImage = convertToTrueColorCopy($imageResource);
imagedestroy($imageResource);

$resizedImage = constrainImageSize($normalizedImage, 640);
if ($resizedImage !== $normalizedImage) {
    imagedestroy($normalizedImage);
}

$currentUser = pemohon_current_user();
$userId = (int) ($currentUser['id'] ?? 0);

if ($userId <= 0) {
    imagedestroy($resizedImage);
    http_response_code(500);
    echo json_encode([
        'message' => 'Tidak dapat mengenal pasti pengguna.',
    ]);
    exit;
}

$profile = pemohon_fetch_profile($userId) ?? [];
$relativeDirectory = 'userfiles/pemohon/' . $userId;
$relativeFilename = 'profile-' . date('Ymd-His') . '-' . bin2hex(random_bytes(4)) . '.jpg';
$relativePath = $relativeDirectory . '/' . $relativeFilename;
$absoluteDirectory = dirname(__DIR__) . '/' . $relativeDirectory;
$absolutePath = dirname(__DIR__) . '/' . $relativePath;

if (!is_dir($absoluteDirectory) && !mkdir($absoluteDirectory, 0775, true) && !is_dir($absoluteDirectory)) {
    imagedestroy($resizedImage);
    http_response_code(500);
    echo json_encode([
        'message' => 'Tidak dapat menyediakan direktori simpanan foto.',
    ]);
    exit;
}

$saveResult = imagejpeg($resizedImage, $absolutePath, 92);
imagedestroy($resizedImage);

if (!$saveResult) {
    @unlink($absolutePath);
    http_response_code(500);
    echo json_encode([
        'message' => 'Gagal menyimpan foto profil.',
    ]);
    exit;
}

chmod($absolutePath, 0664);

$db = pemohon_db();
$db->begin_transaction();

try {
    $updateStmt = $db->prepare('UPDATE applicants SET photo_path = ? WHERE user_id = ?');
    if (!$updateStmt) {
        throw new RuntimeException('Gagal menyediakan kemaskini foto: ' . $db->error);
    }
    $updateStmt->bind_param('si', $relativePath, $userId);
    $updateStmt->execute();
    $affected = $updateStmt->affected_rows;
    $updateStmt->close();

    if ($affected === 0) {
        $insertStmt = $db->prepare('INSERT INTO applicants (user_id, photo_path) VALUES (?, ?)');
        if (!$insertStmt) {
            throw new RuntimeException('Gagal menyediakan simpanan foto: ' . $db->error);
        }
        $insertStmt->bind_param('is', $userId, $relativePath);
        $insertStmt->execute();
        $insertStmt->close();
    }

    $db->commit();
} catch (Throwable $throwable) {
    $db->rollback();
    @unlink($absolutePath);
    http_response_code(500);
    echo json_encode([
        'message' => 'Tidak dapat mengemaskini foto profil.',
        'detail' => $throwable->getMessage(),
    ]);
    exit;
}

if (!empty($profile['photo_path'])) {
    $previousPhoto = dirname(__DIR__) . '/' . ltrim((string) $profile['photo_path'], '/');
    if (is_file($previousPhoto) && strpos(realpath($previousPhoto) ?: '', realpath(dirname(__DIR__) . '/userfiles') ?: '') === 0) {
        @unlink($previousPhoto);
    }
}

$photoUrl = pemohon_asset($relativePath) . '?v=' . urlencode((string) time());

$_SESSION['pemohon_user']['photo_path'] = $relativePath;

echo json_encode([
    'message' => 'Foto profil berjaya dikemaskini.',
    'photoUrl' => $photoUrl,
    'photoPath' => $relativePath,
    'storedBytes' => is_file($absolutePath) ? filesize($absolutePath) : 0,
]);

exit;

/**
 * @param resource $image
 */
/**
 * Creates a copy of the given GD image resource with a white background and true-color palette.
 *
 * @param resource $image
 *
 * @return resource
 */
function convertToTrueColorCopy($image)
{
    $width = imagesx($image);
    $height = imagesy($image);

    $trueColor = imagecreatetruecolor($width, $height);
    imagealphablending($trueColor, true);
    $white = imagecolorallocate($trueColor, 255, 255, 255);
    imagefilledrectangle($trueColor, 0, 0, $width, $height, $white);
    imagecopy($trueColor, $image, 0, 0, 0, 0, $width, $height);

    return $trueColor;
}

/**
 * @param resource $image
 *
 * @return resource
 */
/**
 * Downsizes the image if its largest dimension exceeds the configured maximum.
 *
 * @param resource $image
 *
 * @return resource
 */
function constrainImageSize($image, int $maxSize)
{
    $width = imagesx($image);
    $height = imagesy($image);

    $scale = min(1.0, $maxSize / max($width, $height));
    if ($scale >= 1.0) {
        return $image;
    }

    $targetWidth = (int) round($width * $scale);
    $targetHeight = (int) round($height * $scale);

    $resized = imagecreatetruecolor($targetWidth, $targetHeight);
    imagealphablending($resized, true);
    imagecopyresampled($resized, $image, 0, 0, 0, 0, $targetWidth, $targetHeight, $width, $height);

    return $resized;
}

/**
 * Loads an image resource from disk based on mime type.
 *
 * @return resource|null
 */
function loadImageResource(string $path, string $mimeType)
{
    switch ($mimeType) {
        case 'image/jpeg':
            return @imagecreatefromjpeg($path);
        case 'image/png':
            return @imagecreatefrompng($path);
        default:
            return null;
    }
}

/**
 * @param resource $image
 */
/**
 * Adjusts the orientation of JPEG images based on EXIF metadata.
 *
 * @param resource $image
 *
 * @return resource
 */
function correctImageOrientation($image, string $path, string $mimeType)
{
    if ($mimeType !== 'image/jpeg' || !function_exists('exif_read_data')) {
        return $image;
    }

    $exif = @exif_read_data($path);
    if (!is_array($exif) || !isset($exif['Orientation'])) {
        return $image;
    }

    $orientation = (int) $exif['Orientation'];
    if ($orientation < 2 || $orientation > 8) {
        return $image;
    }

    switch ($orientation) {
        case 2:
            imageflip($image, IMG_FLIP_HORIZONTAL);
            break;
        case 3:
            $rotated = imagerotate($image, 180, 0);
            if ($rotated !== false) {
                imagedestroy($image);
                $image = $rotated;
            }
            break;
        case 4:
            imageflip($image, IMG_FLIP_VERTICAL);
            break;
        case 5:
            $rotated = imagerotate($image, -90, 0);
            if ($rotated !== false) {
                imageflip($rotated, IMG_FLIP_HORIZONTAL);
                imagedestroy($image);
                $image = $rotated;
            }
            break;
        case 6:
            $rotated = imagerotate($image, -90, 0);
            if ($rotated !== false) {
                imagedestroy($image);
                $image = $rotated;
            }
            break;
        case 7:
            $rotated = imagerotate($image, 90, 0);
            if ($rotated !== false) {
                imageflip($rotated, IMG_FLIP_HORIZONTAL);
                imagedestroy($image);
                $image = $rotated;
            }
            break;
        case 8:
            $rotated = imagerotate($image, 90, 0);
            if ($rotated !== false) {
                imagedestroy($image);
                $image = $rotated;
            }
            break;
        default:
            break;
    }

    return $image;
}

/**
 * @param resource $image
 */
