<?php

declare(strict_types=1);

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Request;
use App\Core\Response;
use App\Core\Session;
use mysqli;
use RuntimeException;

class ApplicantAccountController extends Controller
{
    private mysqli $db;

    public function __construct(Request $request, Response $response, Session $session)
    {
        parent::__construct($request, $response, $session);
        $this->db = Database::connection();
    }

    public function index(): void
    {
        $applicants = $this->fetchApplicants();

        $this->view('applicants.account_registration', [
            'title' => 'Pendaftaran Akaun Pemohon',
            'applicants' => $applicants,
            'statuses' => $this->statusLabels(),
            'pageScripts' => $this->pageScripts(),
        ]);
    }

    public function store(): void
    {
        $session = $this->session;
        $data = [
            'name' => trim((string) $this->request->input('name', '')),
            'email' => strtolower(trim((string) $this->request->input('email', ''))),
            'nric' => preg_replace('/\D+/', '', (string) $this->request->input('nric', '')),
            'phone' => trim((string) $this->request->input('phone', '')),
            'status' => $this->normalizeStatus((string) $this->request->input('status', 'pending')),
            'password' => (string) $this->request->input('password', ''),
            'password_confirmation' => (string) $this->request->input('password_confirmation', ''),
        ];

        $errors = $this->validate($data, null);
        if ($errors) {
            if ($this->request->isAjax()) {
                $this->response->json([
                    'success' => false,
                    'message' => 'Sila semak ruangan yang ditandakan.',
                    'errors' => $errors,
                ], 422);
                return;
            }

            $session->flash('error', $errors['_global'] ?? 'Pendaftaran tidak berjaya. Sila semak input anda.');
            $this->response->redirect(route('/modules/applicant/account-registration'));
        }

        $temporaryPhotoPath = null;
        try {
            $temporaryPhotoPath = $this->prepareUploadedPhoto($_FILES['photo'] ?? null);
        } catch (\Throwable $exception) {
            $this->handlePhotoUploadFailure($exception->getMessage());
            return;
        }

        $storedPhotoPath = null;

        try {
            $this->db->begin_transaction();

            $uuid = $this->generateUuid();
            $passwordHash = password_hash($data['password'], PASSWORD_DEFAULT);
            $status = $data['status'];

            $userStmt = $this->db->prepare(
                'INSERT INTO users (uuid, name, email, password, nric, phone, status, force_password_change, created_at, updated_at)
                 VALUES (?, ?, ?, ?, NULLIF(?, \'\'), NULLIF(?, \'\'), ?, 1, NOW(), NOW())'
            );
            if (!$userStmt) {
                throw new RuntimeException('Gagal menyediakan pendaftaran pengguna: ' . $this->db->error);
            }
            $userStmt->bind_param(
                'sssssss',
                $uuid,
                $data['name'],
                $data['email'],
                $passwordHash,
                $data['nric'],
                $data['phone'],
                $status
            );
            $userStmt->execute();
            $userId = (int) $this->db->insert_id;
            $userStmt->close();

            $roleId = $this->fetchApplicantRoleId();
            $roleStmt = $this->db->prepare(
                'INSERT INTO role_user (user_id, role_id, is_primary, assigned_at) VALUES (?, ?, 1, NOW())'
            );
            if (!$roleStmt) {
                throw new RuntimeException('Gagal menyediakan peranan pengguna: ' . $this->db->error);
            }
            $roleStmt->bind_param('ii', $userId, $roleId);
            $roleStmt->execute();
            $roleStmt->close();

            if ($temporaryPhotoPath !== null) {
                $storedPhotoPath = $this->persistPhoto($userId, $temporaryPhotoPath);
                $temporaryPhotoPath = null;
            }

            $applicantStmt = $this->db->prepare(
                'INSERT INTO applicants (user_id, photo_path, nric, created_at, updated_at)
                 VALUES (?, NULLIF(?, \'\'), NULLIF(?, \'\'), NOW(), NOW())'
            );
            if (!$applicantStmt) {
                throw new RuntimeException('Gagal menyediakan maklumat pemohon: ' . $this->db->error);
            }
            $applicantStmt->bind_param('iss', $userId, $storedPhotoPath, $data['nric']);
            $applicantStmt->execute();
            $applicantStmt->close();

            $this->db->commit();
        } catch (\Throwable $exception) {
            $this->db->rollback();
            if ($storedPhotoPath) {
                $this->deletePhoto($storedPhotoPath);
            }
            if ($temporaryPhotoPath) {
                $this->deletePhoto($temporaryPhotoPath);
            }

            if ($this->request->isAjax()) {
                $this->response->json([
                    'success' => false,
                    'message' => 'Pendaftaran tidak dapat disimpan. ' . $exception->getMessage(),
                ], 500);
                return;
            }

            $session->flash('error', 'Pendaftaran tidak dapat disimpan. ' . $exception->getMessage());
            $this->response->redirect(route('/modules/applicant/account-registration'));
        }

        $applicantRecord = $this->fetchApplicant($userId);
        $formattedApplicant = $applicantRecord ? $this->formatApplicant($applicantRecord) : null;

        if ($this->request->isAjax()) {
            $this->response->json([
                'success' => true,
                'message' => 'Akaun pemohon berjaya didaftarkan.',
                'applicant' => $formattedApplicant,
                'mode' => 'create',
            ]);
            return;
        }

        $session->flash('success', 'Akaun pemohon berjaya didaftarkan.');
        $this->response->redirect(route('/modules/applicant/account-registration'));
    }

    public function update(): void
    {
        $session = $this->session;
        $userId = (int) $this->request->input('user_id', 0);
        if ($userId <= 0) {
            if ($this->request->isAjax()) {
                $this->response->json([
                    'success' => false,
                    'message' => 'Pemohon tidak sah.',
                ], 422);
                return;
            }

            $session->flash('error', 'Pemohon tidak sah.');
            $this->response->redirect(route('/modules/applicant/account-registration'));
        }

        $existing = $this->fetchApplicant($userId);
        if ($existing === null) {
            if ($this->request->isAjax()) {
                $this->response->json([
                    'success' => false,
                    'message' => 'Rekod pemohon tidak ditemui.',
                ], 404);
                return;
            }

            $session->flash('error', 'Rekod pemohon tidak ditemui.');
            $this->response->redirect(route('/modules/applicant/account-registration'));
        }

        $data = [
            'name' => trim((string) $this->request->input('name', '')),
            'email' => strtolower(trim((string) $this->request->input('email', ''))),
            'nric' => preg_replace('/\D+/', '', (string) $this->request->input('nric', '')),
            'phone' => trim((string) $this->request->input('phone', '')),
            'status' => $this->normalizeStatus((string) $this->request->input('status', 'pending')),
            'password' => (string) $this->request->input('password', ''),
            'password_confirmation' => (string) $this->request->input('password_confirmation', ''),
        ];

        $errors = $this->validate($data, $userId);
        if ($errors) {
            if ($this->request->isAjax()) {
                $this->response->json([
                    'success' => false,
                    'message' => 'Sila semak ruangan yang ditandakan.',
                    'errors' => $errors,
                ], 422);
                return;
            }

            $session->flash('error', $errors['_global'] ?? 'Kemaskini tidak berjaya. Sila semak input anda.');
            $this->response->redirect(route('/modules/applicant/account-registration'));
        }

        $temporaryPhotoPath = null;
        try {
            $temporaryPhotoPath = $this->prepareUploadedPhoto($_FILES['photo'] ?? null);
        } catch (\Throwable $exception) {
            $this->handlePhotoUploadFailure($exception->getMessage());
            return;
        }

        $oldPhotoPath = $existing['photo_path'] ?? null;
        $newPhotoPath = $oldPhotoPath;
        $photoUpdated = false;

        try {
            $this->db->begin_transaction();

            $passwordHash = null;
            if ($data['password'] !== '') {
                $passwordHash = password_hash($data['password'], PASSWORD_DEFAULT);
            }

            $userSql = 'UPDATE users SET name = ?, email = ?, nric = NULLIF(?, \'\'), phone = NULLIF(?, \'\'), status = ?, updated_at = NOW()';
            $userParams = [
                $data['name'],
                $data['email'],
                $data['nric'],
                $data['phone'],
                $data['status'],
            ];
            $userTypes = 'sssss';

            if ($passwordHash !== null) {
                $userSql .= ', password = ?, force_password_change = 1';
                $userParams[] = $passwordHash;
                $userTypes .= 's';
            }

            $userSql .= ' WHERE id = ? LIMIT 1';
            $userParams[] = $userId;
            $userTypes .= 'i';

            $userStmt = $this->db->prepare($userSql);
            if (!$userStmt) {
                throw new RuntimeException('Gagal menyediakan kemaskini pengguna: ' . $this->db->error);
            }
            $userStmt->bind_param($userTypes, ...$userParams);
            $userStmt->execute();
            $userStmt->close();

            if ($temporaryPhotoPath !== null) {
                $newPhotoPath = $this->persistPhoto($userId, $temporaryPhotoPath);
                $temporaryPhotoPath = null;
                $photoUpdated = true;
            }

            $photoParam = $newPhotoPath ?? '';
            $nricParam = $data['nric'];

            if (!empty($existing['applicant_id'])) {
                $applicantStmt = $this->db->prepare(
                    'UPDATE applicants SET photo_path = NULLIF(?, \'\'), nric = NULLIF(?, \'\'), updated_at = NOW() WHERE user_id = ? LIMIT 1'
                );
                if (!$applicantStmt) {
                    throw new RuntimeException('Gagal mengemaskini maklumat pemohon: ' . $this->db->error);
                }
                $applicantStmt->bind_param('ssi', $photoParam, $nricParam, $userId);
            } else {
                $applicantStmt = $this->db->prepare(
                    'INSERT INTO applicants (user_id, photo_path, nric, created_at, updated_at) VALUES (?, NULLIF(?, \'\'), NULLIF(?, \'\'), NOW(), NOW())'
                );
                if (!$applicantStmt) {
                    throw new RuntimeException('Gagal menambah maklumat pemohon: ' . $this->db->error);
                }
                $applicantStmt->bind_param('iss', $userId, $photoParam, $nricParam);
            }
            $applicantStmt->execute();
            $applicantStmt->close();

            $this->db->commit();
        } catch (\Throwable $exception) {
            $this->db->rollback();
            if ($photoUpdated && $newPhotoPath) {
                $this->deletePhoto($newPhotoPath);
            }
            if ($temporaryPhotoPath) {
                $this->deletePhoto($temporaryPhotoPath);
            }

            if ($this->request->isAjax()) {
                $this->response->json([
                    'success' => false,
                    'message' => 'Kemaskini tidak dapat disimpan. ' . $exception->getMessage(),
                ], 500);
                return;
            }

            $session->flash('error', 'Kemaskini tidak dapat disimpan. ' . $exception->getMessage());
            $this->response->redirect(route('/modules/applicant/account-registration'));
        }

        if ($photoUpdated && $oldPhotoPath && $oldPhotoPath !== $newPhotoPath) {
            $this->deletePhoto($oldPhotoPath);
        }

        $applicantRecord = $this->fetchApplicant($userId);
        $formattedApplicant = $applicantRecord ? $this->formatApplicant($applicantRecord) : null;

        if ($this->request->isAjax()) {
            $this->response->json([
                'success' => true,
                'message' => 'Maklumat pemohon berjaya dikemaskini.',
                'applicant' => $formattedApplicant,
                'mode' => 'update',
            ]);
            return;
        }

        $session->flash('success', 'Maklumat pemohon berjaya dikemaskini.');
        $this->response->redirect(route('/modules/applicant/account-registration'));
    }

    private function fetchApplicants(): array
    {
        $sql = <<<SQL
            SELECT
                u.id,
                u.name,
                u.email,
                u.nric,
                u.phone,
                u.status,
                u.created_at,
                u.last_login_at,
                a.id AS applicant_id,
                a.photo_path
            FROM users u
            INNER JOIN role_user ru ON ru.user_id = u.id AND ru.is_primary = 1
            INNER JOIN roles r ON r.id = ru.role_id
            LEFT JOIN applicants a ON a.user_id = u.id
            WHERE r.name = 'applicant'
            ORDER BY u.created_at DESC
        SQL;

        $result = Database::query($sql);
        $rows = [];
        while ($row = $result->fetch_assoc()) {
            $rows[] = $row;
        }
        $result->free();

        return $rows;
    }

    private function fetchApplicant(int $userId): ?array
    {
        $sql = <<<SQL
            SELECT
                u.id,
                u.name,
                u.email,
                u.nric,
                u.phone,
                u.status,
                u.created_at,
                u.updated_at,
                u.last_login_at,
                a.id AS applicant_id,
                a.photo_path
            FROM users u
            LEFT JOIN applicants a ON a.user_id = u.id
            WHERE u.id = ?
            LIMIT 1
        SQL;

        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new RuntimeException('Gagal mendapatkan maklumat pemohon: ' . $this->db->error);
        }

        $stmt->bind_param('i', $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        $record = $result->fetch_assoc() ?: null;
        $stmt->close();

        return $record ?: null;
    }

    private function fetchApplicantRoleId(): int
    {
        $result = Database::query('SELECT id FROM roles WHERE name = ? LIMIT 1', 's', ['applicant']);
        $role = $result->fetch_assoc();
        $result->free();

        if (!$role) {
            throw new RuntimeException('Peranan applicant tidak wujud.');
        }

        return (int) $role['id'];
    }

    private function validate(array $data, ?int $existingId): array
    {
        $errors = [];

        if ($data['name'] === '') {
            $errors['name'] = 'Nama penuh diperlukan.';
        }

        if ($data['email'] === '' || !filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = 'Alamat emel tidak sah.';
        }

        if ($data['nric'] === '' || !preg_match('/^\d{12}$/', $data['nric'])) {
            $errors['nric'] = 'No. Kad Pengenalan mestilah 12 digit.';
        }

        if ($existingId === null) {
            if ($data['password'] === '' || strlen($data['password']) < 8) {
                $errors['password'] = 'Kata laluan mesti sekurang-kurangnya 8 aksara.';
            } elseif ($data['password'] !== $data['password_confirmation']) {
                $errors['password_confirmation'] = 'Sahkan kata laluan tidak sepadan.';
            }
        } else {
            if ($data['password'] !== '') {
                if (strlen($data['password']) < 8) {
                    $errors['password'] = 'Kata laluan mesti sekurang-kurangnya 8 aksara.';
                } elseif ($data['password'] !== $data['password_confirmation']) {
                    $errors['password_confirmation'] = 'Sahkan kata laluan tidak sepadan.';
                }
            } elseif ($data['password_confirmation'] !== '') {
                $errors['password_confirmation'] = 'Sahkan kata laluan tidak diperlukan jika tidak menukar kata laluan.';
            }
        }

        if ($data['phone'] !== '' && !preg_match('/^[0-9+\-\s]{6,20}$/', $data['phone'])) {
            $errors['phone'] = 'No. telefon tidak sah.';
        }

        if ($data['email'] !== '') {
            if ($existingId === null) {
                $emailExists = $this->exists('SELECT 1 FROM users WHERE email = ? LIMIT 1', 's', [$data['email']]);
            } else {
                $emailExists = $this->exists(
                    'SELECT 1 FROM users WHERE email = ? AND id <> ? LIMIT 1',
                    'si',
                    [$data['email'], $existingId]
                );
            }
            if ($emailExists) {
                $errors['email'] = 'Alamat emel sudah digunakan.';
            }
        }

        if ($data['nric'] !== '') {
            if ($existingId === null) {
                $nricExists = $this->exists('SELECT 1 FROM users WHERE nric = ? LIMIT 1', 's', [$data['nric']]);
            } else {
                $nricExists = $this->exists(
                    'SELECT 1 FROM users WHERE nric = ? AND id <> ? LIMIT 1',
                    'si',
                    [$data['nric'], $existingId]
                );
            }
            if ($nricExists) {
                $errors['nric'] = 'No. Kad Pengenalan sudah wujud.';
            }
        }

        if (!in_array($data['status'], ['pending', 'active', 'suspended', 'deactivated'], true)) {
            $errors['status'] = 'Status akaun tidak sah.';
        }

        if ($errors) {
            $errors['_global'] = 'Sila semak ruangan yang mempunyai ralat.';
        }

        return $errors;
    }

    private function exists(string $sql, string $types, array $params): bool
    {
        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new RuntimeException('Gagal menyediakan semakan duplikasi: ' . $this->db->error);
        }

        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        $stmt->store_result();
        $count = $stmt->num_rows;
        $stmt->close();

        return $count > 0;
    }

    private function storePhoto(array $file): string
    {
        if ($file['size'] > 5 * 1024 * 1024) {
            throw new RuntimeException('Saiz foto melebihi had 5MB.');
        }

        $finfo = new \finfo(FILEINFO_MIME_TYPE);
        $mime = $finfo->file($file['tmp_name']);
        $extension = match ($mime) {
            'image/jpeg' => 'jpg',
            'image/png' => 'png',
            default => null,
        };

        if ($extension === null) {
            throw new RuntimeException('Format foto tidak disokong. Hanya JPEG atau PNG.');
        }

        $directory = dirname(__DIR__, 3) . '/userfiles/tmp';
        if (!is_dir($directory) && !mkdir($directory, 0775, true) && !is_dir($directory)) {
            throw new RuntimeException('Tidak dapat menyediakan direktori foto.');
        }

        $filename = 'profile-' . date('Ymd-His') . '-' . bin2hex(random_bytes(4)) . '.' . $extension;
        $relativePath = 'userfiles/tmp/' . $filename;
        $absolutePath = dirname(__DIR__, 3) . '/' . $relativePath;

        if (!move_uploaded_file($file['tmp_name'], $absolutePath)) {
            throw new RuntimeException('Gagal menyimpan foto profil.');
        }

        return $relativePath;
    }

    private function persistPhoto(int $userId, string $temporaryPath): string
    {
        $baseDir = dirname(__DIR__, 3);
        $normalizedTemp = ltrim(str_replace('\\', '/', $temporaryPath), '/');
        $tempAbsolute = $baseDir . '/' . $normalizedTemp;

        if (!is_file($tempAbsolute)) {
            throw new RuntimeException('Fail foto sementara tidak ditemui.');
        }

        $targetDirectory = $baseDir . '/userfiles/pemohon/' . $userId;
        if (!is_dir($targetDirectory) && !mkdir($targetDirectory, 0775, true) && !is_dir($targetDirectory)) {
            throw new RuntimeException('Tidak dapat menyediakan direktori foto pemohon.');
        }

        $extension = pathinfo($tempAbsolute, PATHINFO_EXTENSION) ?: 'jpg';
        $finalFilename = 'profile-' . date('Ymd-His') . '-' . bin2hex(random_bytes(4)) . '.' . $extension;
        $relativeFinalPath = 'userfiles/pemohon/' . $userId . '/' . $finalFilename;
        $absoluteFinalPath = $baseDir . '/' . $relativeFinalPath;

        if (!rename($tempAbsolute, $absoluteFinalPath)) {
            throw new RuntimeException('Gagal memindahkan foto ke direktori pemohon.');
        }

        @chmod($absoluteFinalPath, 0664);

        return $relativeFinalPath;
    }

    private function deletePhoto(string $relativePath): void
    {
        $absolute = dirname(__DIR__, 3) . '/' . ltrim($relativePath, '/');
        if (is_file($absolute)) {
            @unlink($absolute);
        }
    }

    private function prepareUploadedPhoto(?array $file): ?string
    {
        if (!$file || !is_array($file) || !array_key_exists('error', $file)) {
            return null;
        }

        $error = (int) ($file['error'] ?? UPLOAD_ERR_NO_FILE);
        if ($error === UPLOAD_ERR_NO_FILE) {
            return null;
        }

        if ($error !== UPLOAD_ERR_OK) {
            throw new RuntimeException($this->photoUploadErrorMessage($error));
        }

        if (empty($file['tmp_name'])) {
            throw new RuntimeException('Fail foto tidak sah atau tidak lengkap.');
        }

        return $this->storePhoto($file);
    }

    private function handlePhotoUploadFailure(string $message): void
    {
        $cleanMessage = trim($message);
        $fullMessage = 'Foto tidak dapat diproses. ' . ($cleanMessage !== '' ? $cleanMessage : 'Sila cuba semula.');

        if ($this->request->isAjax()) {
            $this->response->json([
                'success' => false,
                'message' => $fullMessage,
                'errors' => [
                    'photo' => $fullMessage,
                ],
            ], 422);
        }

        $this->session->flash('error', $fullMessage);
        $this->response->redirect(route('/modules/applicant/account-registration'));
    }

    private function photoUploadErrorMessage(int $errorCode): string
    {
        return match ($errorCode) {
            UPLOAD_ERR_INI_SIZE, UPLOAD_ERR_FORM_SIZE => 'Saiz foto melebihi had ' . $this->uploadMaxFilesizeHuman() . '.',
            UPLOAD_ERR_PARTIAL => 'Foto hanya dimuat naik sebahagian. Sila cuba semula.',
            UPLOAD_ERR_NO_TMP_DIR => 'Direktori sementara untuk foto tidak tersedia di pelayan.',
            UPLOAD_ERR_CANT_WRITE => 'Pelayan gagal menyimpan foto ke storan. Sila cuba lagi.',
            UPLOAD_ERR_EXTENSION => 'Muat naik foto dihalang oleh konfigurasi pelayan.',
            default => 'Foto tidak dapat dimuat naik. Sila cuba semula.',
        };
    }

    private function uploadMaxFilesizeHuman(): string
    {
        $value = ini_get('upload_max_filesize');
        if (!is_string($value) || trim($value) === '') {
            return 'yang dibenarkan';
        }

        $value = trim($value);
        $lastChar = strtoupper(substr($value, -1));
        if (ctype_digit($lastChar)) {
            $number = (float) $value;
            $unit = 'B';
        } else {
            $number = (float) substr($value, 0, -1);
            $unit = match ($lastChar) {
                'G' => 'GB',
                'M' => 'MB',
                'K' => 'KB',
                default => 'B',
            };
        }

        if ($number <= 0) {
            return 'yang dibenarkan';
        }

        $formatted = $number == (int) $number
            ? (string) (int) $number
            : rtrim(rtrim(number_format($number, 2), '0'), '.');

        return $formatted . ' ' . $unit;
    }

    public function destroy(): void
    {
        $userId = (int) $this->request->input('user_id', 0);
        if ($userId <= 0) {
            $this->response->json([
                'success' => false,
                'message' => 'Pemohon tidak sah.',
            ], 422);
            return;
        }

        $applicant = $this->fetchApplicant($userId);
        if ($applicant === null) {
            $this->response->json([
                'success' => false,
                'message' => 'Rekod pemohon tidak ditemui.',
            ], 404);
            return;
        }

        $photoPath = $applicant['photo_path'] ?? null;

        try {
            $this->db->begin_transaction();

            $deleteStmt = $this->db->prepare('DELETE FROM users WHERE id = ? LIMIT 1');
            if (!$deleteStmt) {
                throw new RuntimeException('Gagal menyediakan penghapusan pengguna: ' . $this->db->error);
            }
            $deleteStmt->bind_param('i', $userId);
            $deleteStmt->execute();
            $affected = $deleteStmt->affected_rows;
            $deleteStmt->close();

            if ($affected <= 0) {
                throw new RuntimeException('Rekod pengguna tidak ditemui atau telah dipadam.');
            }

            $this->db->commit();
        } catch (\Throwable $exception) {
            $this->db->rollback();
            $this->response->json([
                'success' => false,
                'message' => 'Penghapusan akaun gagal: ' . $exception->getMessage(),
            ], 500);
            return;
        }

        if ($photoPath) {
            $this->deletePhoto($photoPath);
        }

        $this->response->json([
            'success' => true,
            'message' => 'Akaun pemohon berjaya dipadam.',
            'removed_id' => $userId,
        ]);
    }

    private function generateUuid(): string
    {
        $data = random_bytes(16);

        $data[6] = chr((ord($data[6]) & 0x0f) | 0x40);
        $data[8] = chr((ord($data[8]) & 0x3f) | 0x80);

        return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
    }

    private function statusLabels(): array
    {
        return [
            'pending' => 'Menunggu Pengesahan',
            'active' => 'Aktif',
            'suspended' => 'Digantung',
            'deactivated' => 'Dinyahaktifkan',
        ];
    }

    private function formatApplicant(array $record): array
    {
        $photoPath = $record['photo_path'] ?? '';
        if ($photoPath) {
            $photoUrl = '/' . ltrim($photoPath, '/');
            $photoUrl = $photoUrl . '?v=' . urlencode((string) time());
        } else {
            $photoUrl = asset('assets/images/users/1.jpg');
        }

        $registeredRaw = $record['created_at'] ?? null;
        $registeredDisplay = $registeredRaw
            ? date('d/m/Y H:i', strtotime((string) $registeredRaw))
            : '-';

        $lastLoginRaw = $record['last_login_at'] ?? null;
        $lastLoginDisplay = $lastLoginRaw
            ? date('d/m/Y H:i', strtotime((string) $lastLoginRaw))
            : '-';

        $statusKey = $record['status'] ?? 'pending';
        $statusBadgeClass = match ($statusKey) {
            'active' => 'badge bg-success',
            'suspended' => 'badge bg-warning',
            'deactivated' => 'badge bg-secondary',
            default => 'badge bg-info',
        };
        $statusLabels = $this->statusLabels();
        $statusLabel = $statusLabels[$statusKey] ?? ucfirst((string) $statusKey);

        return [
            'id' => (int) ($record['id'] ?? 0),
            'name' => (string) ($record['name'] ?? ''),
            'email' => (string) ($record['email'] ?? ''),
            'nric' => (string) ($record['nric'] ?? ''),
            'phone' => (string) ($record['phone'] ?? ''),
            'status' => $statusKey,
            'statusLabel' => $statusLabel,
            'statusBadgeClass' => $statusBadgeClass,
            'photoUrl' => $photoUrl,
            'photoPath' => (string) ($record['photo_path'] ?? ''),
            'registeredAtDisplay' => $registeredDisplay,
            'registeredAtRaw' => $registeredRaw,
            'lastLoginDisplay' => $lastLoginDisplay,
            'lastLoginRaw' => $lastLoginRaw,
        ];
    }

    private function normalizeStatus(string $status): string
    {
        $allowed = array_keys($this->statusLabels());
        return in_array($status, $allowed, true) ? $status : 'pending';
    }

    private function pageScripts(): string
    {
        $storeUrl = json_encode(
            route('/modules/applicant/account-registration'),
            JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP
        );
        $updateUrl = json_encode(
            route('/modules/applicant/account-registration/update'),
            JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP
        );
        $deleteUrl = json_encode(
            route('/modules/applicant/account-registration/delete'),
            JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP
        );
        $defaultAvatar = json_encode(
            asset('assets/images/users/1.jpg'),
            JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP
        );

        return <<<SCRIPT
<script>
document.addEventListener('DOMContentLoaded', function () {
    var form = document.getElementById('applicant-form');
    if (!form) {
        return;
    }

    var storeUrl = {$storeUrl};
    var updateUrl = {$updateUrl};
    var deleteUrl = {$deleteUrl};
    var defaultAvatar = {$defaultAvatar};

    var alertBox = document.getElementById('applicant-form-alert');
    var submitBtn = document.getElementById('applicant-form-submit');
    var cancelBtn = document.getElementById('applicant-form-cancel');
    var heading = document.getElementById('applicant-form-title');
    var userIdInput = document.getElementById('reg-user-id');
    var passwordInput = document.getElementById('reg-password');
    var passwordConfirmInput = document.getElementById('reg-password-confirm');
    var inputs = {
        name: document.getElementById('reg-name'),
        email: document.getElementById('reg-email'),
        nric: document.getElementById('reg-nric'),
        phone: document.getElementById('reg-phone'),
        status: document.getElementById('reg-status'),
        password: passwordInput,
        password_confirmation: passwordConfirmInput
    };
    var fileInput = document.getElementById('reg-photo');
    var pond = null;
    var filePondRegistered = false;
    var csrfInput = form.querySelector('input[name="_token"]');

    function loadStylesheet(url) {
        return new Promise(function (resolve, reject) {
            if (document.querySelector('link[data-dynamic-style="' + url + '"]') || document.querySelector('link[href="' + url + '"]')) {
                resolve();
                return;
            }
            var link = document.createElement('link');
            link.rel = 'stylesheet';
            link.href = url;
            link.setAttribute('data-dynamic-style', url);
            link.onload = function () { resolve(); };
            link.onerror = function (error) { console.error('Gagal memuatkan gaya:', url, error); resolve(); };
            document.head.appendChild(link);
        });
    }

    function loadScript(url) {
        return new Promise(function (resolve, reject) {
            var existing = document.querySelector('script[data-dynamic-script="' + url + '"]') || document.querySelector('script[src="' + url + '"]');
            if (existing) {
                if (existing.dataset.loaded === 'true') {
                    resolve();
                    return;
                }
                existing.addEventListener('load', function () { resolve(); }, { once: true });
                existing.addEventListener('error', function () { reject(new Error('Script failed: ' + url)); }, { once: true });
                return;
            }

            var script = document.createElement('script');
            script.src = url;
            script.async = true;
            script.setAttribute('data-dynamic-script', url);
            script.onload = function () {
                script.dataset.loaded = 'true';
                resolve();
            };
            script.onerror = function () {
                reject(new Error('Script failed: ' + url));
            };
            document.head.appendChild(script);
        });
    }

    function loadScriptWithFallback(urls) {
        var index = 0;
        function attempt() {
            if (index >= urls.length) {
                return Promise.reject(new Error('Semua percubaan memuatkan skrip gagal.'));
            }
            var url = urls[index++];
            return loadScript(url).catch(function (error) {
                console.warn('Tidak dapat memuatkan skrip, cuba pautan lain:', url, error);
                return attempt();
            });
        }
        return attempt();
    }

    function ensureSweetAlert() {
        if (window.Swal) {
            return Promise.resolve();
        }
        return loadScriptWithFallback([
            'https://cdn.jsdelivr.net/npm/sweetalert2@11',
            'https://unpkg.com/sweetalert2@11'
        ]).catch(function () {
            return Promise.resolve();
        });
    }

    function ensureFilePondAssets() {
        var styleUrls = [
            'https://unpkg.com/filepond@^4/dist/filepond.min.css',
            'https://unpkg.com/filepond-plugin-image-preview@^4/dist/filepond-plugin-image-preview.min.css'
        ];
        var scriptGroups = [
            ['https://unpkg.com/filepond@^4/dist/filepond.min.js', 'https://cdn.jsdelivr.net/npm/filepond@4/dist/filepond.min.js'],
            ['https://unpkg.com/filepond-plugin-file-validate-type@^1/dist/filepond-plugin-file-validate-type.min.js', 'https://cdn.jsdelivr.net/npm/filepond-plugin-file-validate-type@1/dist/filepond-plugin-file-validate-type.min.js'],
            ['https://unpkg.com/filepond-plugin-file-validate-size@^2/dist/filepond-plugin-file-validate-size.min.js', 'https://cdn.jsdelivr.net/npm/filepond-plugin-file-validate-size@2/dist/filepond-plugin-file-validate-size.min.js'],
            ['https://unpkg.com/filepond-plugin-image-preview@^4/dist/filepond-plugin-image-preview.min.js', 'https://cdn.jsdelivr.net/npm/filepond-plugin-image-preview@4/dist/filepond-plugin-image-preview.min.js']
        ];

        return Promise.all(styleUrls.map(loadStylesheet)).then(function () {
            return scriptGroups.reduce(function (promise, urls) {
                return promise.then(function () {
                    return loadScriptWithFallback(urls).catch(function (error) {
                        console.warn(error.message);
                        return Promise.resolve();
                    });
                });
            }, Promise.resolve());
        });
    }

    function setupFilePond() {
        if (!fileInput) {
            return;
        }
        fileInput.classList.add('filepond');
        ensureFilePondAssets().then(function () {
            if (typeof FilePond === 'undefined') {
                console.warn('FilePond tidak tersedia.');
                return;
            }
            if (!filePondRegistered) {
                var plugins = [];
                if (typeof FilePondPluginFileValidateType !== 'undefined') {
                    plugins.push(FilePondPluginFileValidateType);
                }
                if (typeof FilePondPluginFileValidateSize !== 'undefined') {
                    plugins.push(FilePondPluginFileValidateSize);
                }
                if (typeof FilePondPluginImagePreview !== 'undefined') {
                    plugins.push(FilePondPluginImagePreview);
                }
                if (plugins.length > 0) {
                    FilePond.registerPlugin.apply(FilePond, plugins);
                }
                filePondRegistered = true;
            }
            if (pond) {
                pond.destroy();
            }
            pond = FilePond.create(fileInput, {
                allowMultiple: false,
                instantUpload: false,
                credits: false,
                acceptedFileTypes: ['image/jpeg', 'image/png'],
                maxFileSize: '5MB',
                labelIdle: 'Seret & lepas foto atau <span class="filepond--label-action">pilih</span>',
                imagePreviewHeight: 140,
                stylePanelLayout: 'compact'
            });
        }).catch(function (error) {
            console.error('Gagal menyediakan FilePond:', error);
        });
    }

    setupFilePond();
    ensureSweetAlert();

    var defaultButtonText = submitBtn ? (submitBtn.getAttribute('data-default-text') || submitBtn.textContent) : '';
    var editButtonText = submitBtn ? (submitBtn.getAttribute('data-edit-text') || submitBtn.textContent) : '';
    var loadingText = submitBtn ? (submitBtn.getAttribute('data-loading-text') || 'Memproses...') : '';

    if (heading) {
        if (!heading.dataset.createTitle) {
            heading.dataset.createTitle = heading.textContent;
        }
        if (!heading.dataset.editTitle) {
            heading.dataset.editTitle = heading.getAttribute('data-edit-title') || 'Kemaskini Maklumat Pemohon';
        }
    }
    if (inputs.status && !inputs.status.dataset.defaultValue) {
        inputs.status.dataset.defaultValue = inputs.status.value || 'pending';
    }

    var dataTable = null;
    if (window.jQuery && typeof window.jQuery.fn.DataTable === 'function') {
        var tableEl = window.jQuery('#applicants-table');
        if (tableEl.length) {
            if (window.jQuery.fn.DataTable.isDataTable(tableEl)) {
                dataTable = tableEl.DataTable();
            } else {
                dataTable = tableEl.DataTable({
                    responsive: true,
                    pageLength: 10,
                    order: [[5, 'desc']],
                    columnDefs: [
                        { targets: [0, 7], orderable: false, searchable: false },
                        { targets: [0, 4, 7], className: 'text-center align-middle' },
                        { targets: [1, 2, 3, 5, 6], className: 'align-middle' }
                    ]
                });
            }
        }
    }

    function escapeHtml(value) {
        if (value === null || value === undefined) {
            return '';
        }
        return String(value).replace(/[&<>"']/g, function (char) {
            switch (char) {
                case '&': return '&amp;';
                case '<': return '&lt;';
                case '>': return '&gt;';
                case '"': return '&quot;';
                case '\'': return '&#039;';
                default: return char;
            }
        });
    }

    function showMessage(type, message) {
        if (!alertBox) {
            return;
        }
        alertBox.classList.remove('alert-success', 'alert-danger', 'alert-warning', 'alert-info');
        if (!message) {
            alertBox.classList.add('d-none');
            alertBox.textContent = '';
            return;
        }
        alertBox.classList.remove('d-none');
        alertBox.classList.add('alert', 'alert-' + type);
        alertBox.textContent = message;
    }

    function clearFieldErrors() {
        Object.keys(inputs).forEach(function (key) {
            var field = inputs[key];
            if (!field) {
                return;
            }
            field.classList.remove('is-invalid');
            var group = field.closest('.mb-3');
            if (group) {
                var feedback = group.querySelector('.invalid-feedback');
                if (feedback) {
                    feedback.textContent = '';
                }
            }
        });
    }

    function setFieldError(fieldName, message) {
        var field = inputs[fieldName];
        if (!field) {
            return;
        }
        var group = field.closest('.mb-3');
        if (message) {
            field.classList.add('is-invalid');
            if (group) {
                var feedback = group.querySelector('.invalid-feedback');
                if (feedback) {
                    feedback.textContent = message;
                }
            }
        } else {
            field.classList.remove('is-invalid');
            if (group) {
                var feedback = group.querySelector('.invalid-feedback');
                if (feedback) {
                    feedback.textContent = '';
                }
            }
        }
    }

    function applyErrors(errors) {
        clearFieldErrors();
        if (!errors) {
            return;
        }
        var fieldMessages = [];
        Object.keys(errors).forEach(function (key) {
            if (key === '_global') {
                return;
            }
            fieldMessages.push(errors[key]);
            setFieldError(key, errors[key]);
        });
        var message = errors._global || fieldMessages.join(' ');
        if (message) {
            showMessage('danger', message);
        }
    }

    function setSubmitting(isSubmitting) {
        if (!submitBtn) {
            return;
        }
        if (isSubmitting) {
            submitBtn.disabled = true;
            submitBtn.dataset.previousText = submitBtn.textContent;
            submitBtn.textContent = loadingText || submitBtn.textContent;
        } else {
            submitBtn.disabled = false;
            if (
                submitBtn.dataset.previousText &&
                submitBtn.textContent === (loadingText || submitBtn.dataset.previousText)
            ) {
                submitBtn.textContent = submitBtn.dataset.previousText;
            }
            delete submitBtn.dataset.previousText;
        }
    }

    function createRowElement(html) {
        var wrapper = document.createElement('tbody');
        wrapper.innerHTML = html.trim();
        return wrapper.firstElementChild;
    }

    function buildRowRepresentation(applicant) {
        var rowId = 'applicant-row-' + (applicant.id || '');
        var statusBadgeClass = applicant.statusBadgeClass || 'badge bg-info';
        var statusLabel = applicant.statusLabel || (applicant.status ? applicant.status.charAt(0).toUpperCase() + applicant.status.slice(1) : '-');
        var daftarDisplay = applicant.registeredAtDisplay || '-';
        var lastLoginDisplay = applicant.lastLoginDisplay || '-';
        var daftarOrder = applicant.registeredAtRaw || '';
        var lastLoginOrder = applicant.lastLoginRaw || '';
        var photoSrc = applicant.photoUrl || defaultAvatar;
        var applicantJson = escapeHtml(JSON.stringify(applicant));

        var columns = [
            '<td class="text-center align-middle"><img src="' + escapeHtml(photoSrc) + '" alt="Avatar" class="rounded-circle" style="width:48px;height:48px;object-fit:cover;"></td>',
            '<td class="align-middle">' + escapeHtml(applicant.name || '-') + '</td>',
            '<td class="align-middle">' + escapeHtml(applicant.email || '-') + '</td>',
            '<td class="align-middle">' + escapeHtml(applicant.nric || '-') + '</td>',
            '<td class="text-center align-middle"><span class="' + escapeHtml(statusBadgeClass) + '">' + escapeHtml(statusLabel) + '</span></td>',
            '<td class="align-middle" data-order="' + escapeHtml(daftarOrder) + '">' + escapeHtml(daftarDisplay) + '</td>',
            '<td class="align-middle" data-order="' + escapeHtml(lastLoginOrder) + '">' + escapeHtml(lastLoginDisplay) + '</td>',
            '<td class="text-center align-middle"><div class="d-flex flex-column gap-2"><button type="button" class="btn btn-sm btn-outline-primary js-edit-applicant" data-applicant="' + applicantJson + '"><i class="fas fa-edit me-1"></i> Kemaskini</button><button type="button" class="btn btn-sm btn-outline-danger js-delete-applicant" data-applicant-id="' + escapeHtml(String(applicant.id || '')) + '" data-applicant-name="' + escapeHtml(applicant.name || '') + '"><i class="fas fa-trash me-1"></i> Padam</button></div></td>'
        ];

        return {
            id: rowId,
            html: '<tr id="' + rowId + '">' + columns.join('') + '</tr>'
        };
    }

    function upsertRow(applicant) {
        if (!applicant || !applicant.id) {
            return;
        }
        var representation = buildRowRepresentation(applicant);
        var existingRow = document.getElementById(representation.id);
        if (dataTable) {
            if (existingRow) {
                dataTable.row(existingRow).remove();
            }
            var newRowElement = createRowElement(representation.html);
            if (!newRowElement) {
                return;
            }
            dataTable.row.add(newRowElement).draw(false);
        } else {
            var tbody = document.querySelector('#applicants-table tbody');
            if (!tbody) {
                return;
            }
            if (existingRow) {
                existingRow.outerHTML = representation.html;
            } else {
                tbody.insertAdjacentHTML('afterbegin', representation.html);
            }
        }
    }

    function removeRowById(applicantId) {
        var rowId = 'applicant-row-' + applicantId;
        var row = document.getElementById(rowId);
        if (!row) {
            return;
        }
        if (dataTable) {
            dataTable.row(row).remove().draw(false);
        } else if (row.parentNode) {
            row.parentNode.removeChild(row);
        }
    }

    function setFormMode(mode, applicant) {
        var isEdit = mode === 'edit';
        form.reset();
        clearFieldErrors();
        if (pond) {
            pond.removeFiles();
        }

        if (userIdInput) {
            userIdInput.value = '';
        }
        if (inputs.status) {
            inputs.status.value = inputs.status.dataset.defaultValue || 'pending';
        }
        if (passwordInput) {
            passwordInput.value = '';
            passwordInput.required = !isEdit;
        }
        if (passwordConfirmInput) {
            passwordConfirmInput.value = '';
            passwordConfirmInput.required = !isEdit;
        }

        if (isEdit && applicant) {
            if (userIdInput) {
                userIdInput.value = applicant.id || '';
            }
            if (inputs.name) inputs.name.value = applicant.name || '';
            if (inputs.email) inputs.email.value = applicant.email || '';
            if (inputs.nric) inputs.nric.value = applicant.nric || '';
            if (inputs.phone) inputs.phone.value = applicant.phone || '';
            if (inputs.status && applicant.status) {
                inputs.status.value = applicant.status;
            }
            if (heading) {
                heading.textContent = heading.dataset.editTitle || 'Kemaskini Maklumat Pemohon';
            }
            if (submitBtn) {
                submitBtn.textContent = editButtonText || submitBtn.textContent;
            }
            if (cancelBtn) {
                cancelBtn.classList.remove('d-none');
            }
            if (inputs.name) {
                inputs.name.focus();
            }
        } else {
            if (heading) {
                heading.textContent = heading.dataset.createTitle || heading.textContent;
            }
            if (submitBtn) {
                submitBtn.textContent = defaultButtonText || submitBtn.textContent;
            }
            if (cancelBtn) {
                cancelBtn.classList.add('d-none');
            }
        }
    }

    setFormMode('create');

    if (cancelBtn) {
        cancelBtn.addEventListener('click', function () {
            setFormMode('create');
        });
    }

    var tableElement = document.getElementById('applicants-table');
    if (tableElement) {
        tableElement.addEventListener('click', function (event) {
            var editButton = event.target.closest('.js-edit-applicant');
            if (editButton) {
                var raw = editButton.getAttribute('data-applicant');
                if (!raw) {
                    return;
                }
                try {
                    var applicant = JSON.parse(raw);
                    showMessage('', '');
                    setFormMode('edit', applicant);
                } catch (error) {
                    console.error('Gagal mengurai data pemohon:', error);
                }
                return;
            }

            var deleteButton = event.target.closest('.js-delete-applicant');
            if (!deleteButton) {
                return;
            }
            var applicantId = Number(deleteButton.getAttribute('data-applicant-id'));
            if (!applicantId) {
                return;
            }
            var applicantName = deleteButton.getAttribute('data-applicant-name') || 'pemohon';

            var performDeletion = function () {
                if (!deleteUrl) {
                    return;
                }
                var formData = new FormData();
                if (csrfInput && csrfInput.value) {
                    formData.append('_token', csrfInput.value);
                }
                formData.append('user_id', String(applicantId));

                deleteButton.disabled = true;

                fetch(deleteUrl, {
                    method: 'POST',
                    credentials: 'same-origin',
                    body: formData,
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest'
                    }
                }).then(function (response) {
                    return response.json().catch(function () { return {}; }).then(function (data) {
                        return { ok: response.ok, status: response.status, body: data };
                    });
                }).then(function (result) {
                    deleteButton.disabled = false;
                    var message = result.body && result.body.message ? result.body.message : 'Permintaan tidak dapat diproses.';
                    if (!result.ok) {
                        if (window.Swal) {
                            Swal.fire({
                                icon: 'error',
                                title: 'Tidak Berjaya',
                                text: message
                            });
                        } else {
                            alert(message);
                        }
                        return;
                    }

                    removeRowById(applicantId);
                    showMessage('success', message);
                }).catch(function () {
                    deleteButton.disabled = false;
                    var fallback = 'Penghapusan tidak dapat diproses sekarang.';
                    if (window.Swal) {
                        Swal.fire({
                            icon: 'error',
                            title: 'Tidak Berjaya',
                            text: fallback
                        });
                    } else {
                        alert(fallback);
                    }
                });
            };

            if (window.Swal) {
                Swal.fire({
                    title: 'Padam Akaun?',
                    text: 'Anda akan memadam akaun bagi ' + applicantName + '. Tindakan ini tidak boleh diundur.',
                    icon: 'warning',
                    showCancelButton: true,
                    confirmButtonText: 'Ya, Padam',
                    cancelButtonText: 'Batal',
                    confirmButtonColor: '#d33'
                }).then(function (result) {
                    if (result.isConfirmed) {
                        performDeletion();
                    }
                });
            } else if (confirm('Padam akaun bagi ' + applicantName + '?')) {
                performDeletion();
            }
        });
    }

    form.addEventListener('submit', function (event) {
        event.preventDefault();
        clearFieldErrors();
        showMessage('', '');

        var isEdit = userIdInput && userIdInput.value !== '';
        var targetUrl = isEdit ? updateUrl : storeUrl;
        if (!targetUrl) {
            return;
        }

        var formData = new FormData(form);
        console.log('Submitting form. FilePond instance:', pond);
        if (pond && typeof pond.getFiles === 'function') {
            var pondFiles = pond.getFiles().filter(function (fileItem) {
                return !!fileItem && fileItem.file instanceof Blob;
            });
            if (pondFiles.length > 0) {
                console.log('FilePond has files:', pondFiles);
                var uploadFile = pondFiles[0].file;
                var uploadName = pondFiles[0].file.name || 'photo.jpg';
                formData.delete('photo');
                formData.append('photo', uploadFile, uploadName);
            }
        }
        setSubmitting(true);

        fetch(targetUrl, {
            method: 'POST',
            body: formData,
            credentials: 'same-origin',
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        }).then(function (response) {
            return response.json()
                .catch(function () { return {}; })
                .then(function (data) {
                    return { ok: response.ok, status: response.status, data: data };
                });
        }).then(function (payload) {
            var data = payload.data || {};
            if (!payload.ok || data.success === false) {
                applyErrors(data.errors || null);
                var message = data.message || 'Permintaan tidak dapat diproses.';
                showMessage('danger', message);
                return;
            }
            applyErrors(null);
            var applicant = data.applicant || null;
            if (applicant) {
                upsertRow(applicant);
            }
            showMessage('success', data.message || 'Maklumat berjaya disimpan.');
            setFormMode('create');
        }).catch(function (error) {
            console.error('Ralat permintaan:', error);
            showMessage('danger', 'Permintaan tidak dapat diproses buat masa ini.');
        }).finally(function () {
            setSubmitting(false);
        });
    });
});
</script>
SCRIPT;
    }
}
