<?php

declare(strict_types=1);

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Request;
use App\Core\Response;
use App\Core\Session;
use mysqli;
use RuntimeException;

class NotificationsController extends Controller
{
    private mysqli $db;

    /**
     * @var array<string, string>
     */
    private array $rolesMap = [
        'applicant' => 'Pemohon',
        'admin' => 'Admin',
        'pentadbir' => 'Pentadbir',
        'pentadbir_agong' => 'Pentadbir Agong',
        'panel' => 'Panel',
    ];

    public function __construct(Request $request, Response $response, Session $session)
    {
        parent::__construct($request, $response, $session);
        $this->db = Database::connection();
    }

    public function index(): void
    {
        $announcements = $this->fetchAnnouncements();
        $this->view('admin.notifications.index', [
            'title' => 'Pengurusan Notifikasi',
            'announcements' => $announcements,
            'rolesMap' => $this->rolesMap,
        ]);
    }

    public function save(): void
    {
        if (!$this->request->isAjax()) {
            $this->response->redirect('/modules/admin/notifications');
        }

        $id = (int) $this->request->input('id', 0);
        $data = [
            'title' => trim((string) $this->request->input('title', '')),
            'body' => trim((string) $this->request->input('body', '')),
            'audience_mode' => (string) $this->request->input('audience_mode', 'all'),
            'target_roles' => $this->normalizeArray($this->request->input('target_roles')),
            'is_active' => (int) $this->request->input('is_active', 0) === 1 ? 1 : 0,
            'published_from' => $this->parseDateTime($this->request->input('published_from')),
            'published_until' => $this->parseDateTime($this->request->input('published_until')),
        ];

        $errors = $this->validateAnnouncement($data);
        if ($errors) {
            $this->response->json([
                'success' => false,
                'message' => 'Sila semak maklumat yang dimasukkan.',
                'errors' => $errors,
            ], 422);
        }

        $user = $this->session->get('user') ?? [];
        $adminId = isset($user['id']) ? (int) $user['id'] : null;
        if ($adminId !== null && $adminId <= 0) {
            $adminId = null;
        }

        $audience = 'all';
        $targetRoles = [];
        if ($data['audience_mode'] === 'all') {
            $audience = 'all';
        } elseif ($data['audience_mode'] === 'applicant' || $data['audience_mode'] === 'admin' || $data['audience_mode'] === 'panel') {
            $audience = $data['audience_mode'];
        } else {
            $audience = 'custom';
            $targetRoles = array_values(array_unique(array_filter($data['target_roles'], function ($role) {
                return isset($this->rolesMap[$role]);
            })));
        }

        $this->db->begin_transaction();
        try {
            if ($id > 0) {
                $stmt = $this->db->prepare(
                    'UPDATE announcements
                     SET title = ?, body = ?, audience = ?, is_active = ?, published_from = ?, published_until = ?, updated_by = ?, updated_at = NOW()
                     WHERE id = ?'
                );
                if (!$stmt) {
                    throw new RuntimeException('Gagal menyediakan kemaskini notifikasi: ' . $this->db->error);
                }
                $stmt->bind_param(
                    'sssisiii',
                    $data['title'],
                    $data['body'],
                    $audience,
                    $data['is_active'],
                    $data['published_from'],
                    $data['published_until'],
                    $adminId,
                    $id
                );
                $stmt->execute();
                $stmt->close();

                $deleteStmt = $this->db->prepare('DELETE FROM announcement_targets WHERE announcement_id = ?');
                if ($deleteStmt) {
                    $deleteStmt->bind_param('i', $id);
                    $deleteStmt->execute();
                    $deleteStmt->close();
                }
            } else {
                $stmt = $this->db->prepare(
                    'INSERT INTO announcements (title, body, audience, is_active, published_from, published_until, created_by, updated_by, created_at, updated_at)
                     VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())'
                );
                if (!$stmt) {
                    throw new RuntimeException('Gagal menyediakan notifikasi baharu: ' . $this->db->error);
                }
                $stmt->bind_param(
                    'sssissii',
                    $data['title'],
                    $data['body'],
                    $audience,
                    $data['is_active'],
                    $data['published_from'],
                    $data['published_until'],
                    $adminId,
                    $adminId
                );
                $stmt->execute();
                $id = (int) $this->db->insert_id;
                $stmt->close();
            }

            if ($audience === 'custom' && $targetRoles !== []) {
                $insertStmt = $this->db->prepare('INSERT INTO announcement_targets (announcement_id, target_role) VALUES (?, ?)');
                if (!$insertStmt) {
                    throw new RuntimeException('Gagal menyimpan sasaran notifikasi: ' . $this->db->error);
                }
                foreach ($targetRoles as $role) {
                    $insertStmt->bind_param('is', $id, $role);
                    $insertStmt->execute();
                }
                $insertStmt->close();
            }

            $this->db->commit();
        } catch (\Throwable $exception) {
            $this->db->rollback();
            $this->response->json([
                'success' => false,
                'message' => 'Operasi tidak dapat diselesaikan: ' . $exception->getMessage(),
            ], 500);
        }

        $announcement = $this->findAnnouncement($id);

        $this->response->json([
            'success' => true,
            'message' => $id > 0 ? 'Notifikasi berjaya dikemaskini.' : 'Notifikasi baharu berjaya ditambah.',
            'announcement' => $announcement,
        ]);
    }

    public function delete(): void
    {
        if (!$this->request->isAjax()) {
            $this->response->redirect('/modules/admin/notifications');
        }

        $id = (int) $this->request->input('id', 0);
        if ($id <= 0) {
            $this->response->json([
                'success' => false,
                'message' => 'Notifikasi tidak sah.',
            ], 422);
        }

        $stmt = $this->db->prepare('DELETE FROM announcements WHERE id = ?');
        if (!$stmt) {
            $this->response->json([
                'success' => false,
                'message' => 'Gagal menghapuskan notifikasi: ' . $this->db->error,
            ], 500);
        }

        $stmt->bind_param('i', $id);
        $stmt->execute();
        $stmt->close();

        $this->response->json([
            'success' => true,
            'message' => 'Notifikasi berjaya dipadam.',
        ]);
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    private function fetchAnnouncements(): array
    {
        $sql = <<<SQL
            SELECT
                a.id,
                a.title,
                a.body,
                a.audience,
                a.is_active,
                a.published_from,
                a.published_until,
                a.created_at,
                a.updated_at,
                a.created_by,
                a.updated_by,
                t.target_role
            FROM announcements a
            LEFT JOIN announcement_targets t ON t.announcement_id = a.id
            ORDER BY a.created_at DESC
        SQL;

        $result = $this->db->query($sql);
        if (!$result) {
            throw new RuntimeException('Gagal mendapatkan senarai notifikasi: ' . $this->db->error);
        }

        $announcements = [];
        while ($row = $result->fetch_assoc()) {
            $id = (int) $row['id'];
            if (!isset($announcements[$id])) {
                $announcements[$id] = [
                    'id' => $id,
                    'title' => (string) $row['title'],
                    'body' => (string) $row['body'],
                    'audience' => (string) $row['audience'],
                    'audience_label' => $this->audienceLabel((string) $row['audience']),
                    'is_active' => (int) $row['is_active'],
                    'published_from' => $this->formatDateTime($row['published_from']),
                    'published_until' => $this->formatDateTime($row['published_until']),
                    'published_from_display' => $this->formatDateTime($row['published_from'], 'd/m/Y H:i'),
                    'published_until_display' => $this->formatDateTime($row['published_until'], 'd/m/Y H:i'),
                    'created_at_display' => $this->formatDateTime($row['created_at'], 'd/m/Y H:i'),
                    'targets' => [],
                ];
            }

            if (!empty($row['target_role'])) {
                $announcements[$id]['targets'][] = (string) $row['target_role'];
            }
        }

        $result->free();

        foreach ($announcements as &$announcement) {
            $uniqueTargets = array_values(array_unique($announcement['targets']));
            $announcement['targets'] = $uniqueTargets;
            if ($announcement['audience'] === 'custom') {
                $labels = array_map(function (string $role): string {
                    return $this->rolesMap[$role] ?? ucfirst($role);
                }, $uniqueTargets);
                $announcement['audience_label'] = $labels ? implode(', ', $labels) : 'Tersuai';
            }
        }
        unset($announcement);

        return array_values($announcements);
    }

    /**
     * @return array<string>
     */
    private function normalizeArray(mixed $value): array
    {
        if (is_array($value)) {
            return array_map('strval', $value);
        }

        if (is_string($value) && $value !== '') {
            return [$value];
        }

        return [];
    }

    /**
     * @param array<string, mixed> $data
     *
     * @return array<string, string>
     */
    private function validateAnnouncement(array $data): array
    {
        $errors = [];

        if ($data['title'] === '') {
            $errors['title'] = 'Tajuk diperlukan.';
        }

        if ($data['body'] === '') {
            $errors['body'] = 'Kandungan notifikasi diperlukan.';
        }

        $audienceMode = $data['audience_mode'];
        $validModes = ['all', 'applicant', 'admin', 'panel', 'custom'];
        if (!in_array($audienceMode, $validModes, true)) {
            $errors['audience_mode'] = 'Pemilihan sasaran tidak sah.';
        }

        if ($audienceMode === 'custom') {
            if ($data['target_roles'] === []) {
                $errors['target_roles'] = 'Sila pilih sekurang-kurangnya satu kumpulan sasaran.';
            } else {
                foreach ($data['target_roles'] as $role) {
                    if (!isset($this->rolesMap[$role])) {
                        $errors['target_roles'] = 'Pemilihan kumpulan sasaran tidak sah.';
                        break;
                    }
                }
            }
        }

        if ($data['published_from'] && $data['published_until']) {
            if (strtotime($data['published_from']) > strtotime($data['published_until'])) {
                $errors['published_until'] = 'Tarikh tamat hendaklah selepas tarikh mula.';
            }
        }

        return $errors;
    }

    /**
     * @return array<string, mixed>|null
     */
    private function findAnnouncement(int $id): ?array
    {
        $sql = <<<SQL
            SELECT
                a.*,
                t.target_role
            FROM announcements a
            LEFT JOIN announcement_targets t ON t.announcement_id = a.id
            WHERE a.id = ?
        SQL;

        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new RuntimeException('Gagal mendapatkan notifikasi: ' . $this->db->error);
        }
        $stmt->bind_param('i', $id);
        $stmt->execute();
        $result = $stmt->get_result();

        $announcement = null;
        while ($row = $result->fetch_assoc()) {
            if ($announcement === null) {
                $announcement = [
                    'id' => (int) $row['id'],
                    'title' => (string) $row['title'],
                    'body' => (string) $row['body'],
                    'audience' => (string) $row['audience'],
                    'audience_label' => $this->audienceLabel((string) $row['audience']),
                    'is_active' => (int) $row['is_active'],
                    'published_from' => $this->formatDateTime($row['published_from']),
                    'published_until' => $this->formatDateTime($row['published_until']),
                    'published_from_display' => $this->formatDateTime($row['published_from'], 'd/m/Y H:i'),
                    'published_until_display' => $this->formatDateTime($row['published_until'], 'd/m/Y H:i'),
                    'created_at_display' => $this->formatDateTime($row['created_at'], 'd/m/Y H:i'),
                    'targets' => [],
                ];
            }

            if (!empty($row['target_role'])) {
                $announcement['targets'][] = (string) $row['target_role'];
            }
        }

        $stmt->close();

        if ($announcement !== null) {
            $announcement['targets'] = array_values(array_unique($announcement['targets']));
            if ($announcement['audience'] === 'custom') {
                $labels = array_map(function (string $role): string {
                    return $this->rolesMap[$role] ?? ucfirst($role);
                }, $announcement['targets']);
                $announcement['audience_label'] = $labels ? implode(', ', $labels) : 'Tersuai';
            }
        }

        return $announcement;
    }

    private function parseDateTime(mixed $value): ?string
    {
        if (!is_string($value) || trim($value) === '') {
            return null;
        }

        $timestamp = strtotime($value);
        if ($timestamp === false) {
            return null;
        }

        return date('Y-m-d H:i:s', $timestamp);
    }

    private function formatDateTime(mixed $value, string $format = 'Y-m-d H:i:s'): ?string
    {
        if (!$value) {
            return null;
        }

        $timestamp = strtotime((string) $value);
        if ($timestamp === false) {
            return null;
        }

        return date($format, $timestamp);
    }

    private function audienceLabel(string $audience): string
    {
        return match ($audience) {
            'all' => 'Semua Pengguna',
            'admin' => 'Semua Admin',
            'applicant' => 'Pemohon',
            'panel' => 'Panel',
            'custom' => 'Tersuai',
            default => ucfirst($audience),
        };
    }
}
