<?php

declare(strict_types=1);

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Request;
use App\Core\Response;
use App\Core\Session;
use App\Services\AuthService;
use App\Services\PasswordResetService;
use RuntimeException;

class PasswordResetController extends Controller
{
    private PasswordResetService $passwordResetService;
    private AuthService $authService;

    public function __construct(Request $request, Response $response, Session $session)
    {
        parent::__construct($request, $response, $session);
        $this->passwordResetService = new PasswordResetService();
        $this->authService = new AuthService($session);
    }

    public function sendTemporaryPassword(): void
    {
        $csrfToken = (string) $this->request->input('_token', '');
        if (!validate_csrf_token($csrfToken)) {
            $this->response->json(['message' => 'Sesi tidak sah. Sila muat semula halaman.'], 419);
        }

        $email = strtolower(trim((string) $this->request->input('email', '')));
        if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $this->response->json([
                'message' => 'Emel tidak sah.',
                'errors' => ['email' => 'Sila masukkan alamat emel yang sah.'],
            ], 422);
        }

        try {
            $this->passwordResetService->requestTemporaryPassword($email);
        } catch (RuntimeException $exception) {
            $this->response->json([
                'message' => 'Kami tidak dapat menghantar emel ketika ini. Sila cuba lagi kemudian.',
            ], 500);
        }

        $this->response->json([
            'message' => 'Jika emel berdaftar, kata laluan sementara telah dihantar.',
        ]);
    }

    public function showResetForm(): void
    {
        $this->view('auth.reset_password', [
            'title' => 'Tetapan Semula Kata Laluan',
            'error' => $this->session->getFlash('error'),
            'success' => $this->session->getFlash('success'),
            'pageScripts' => $this->resetScripts(),
        ], 'guest');
    }

    public function handleReset(): void
    {
        $csrfToken = (string) $this->request->input('_token', '');
        if (!validate_csrf_token($csrfToken)) {
            $this->session->flash('error', 'Sesi tidak sah. Sila cuba lagi.');
            $this->redirect('/reset_password.php');
            return;
        }

        $email = strtolower(trim((string) $this->request->input('email', '')));
        $temporaryPassword = (string) $this->request->input('temporary_password', '');
        $newPassword = (string) $this->request->input('password', '');
        $confirmPassword = (string) $this->request->input('password_confirmation', '');

        if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $this->session->flash('error', 'Sila masukkan alamat emel yang sah.');
            $this->redirect('/reset_password.php');
            return;
        }

        if ($temporaryPassword === '') {
            $this->session->flash('error', 'Sila masukkan kata laluan sementara anda.');
            $this->redirect('/reset_password.php');
            return;
        }

        if (strlen($newPassword) < 8) {
            $this->session->flash('error', 'Kata laluan baharu mestilah sekurang-kurangnya 8 aksara.');
            $this->redirect('/reset_password.php');
            return;
        }

        if ($newPassword !== $confirmPassword) {
            $this->session->flash('error', 'Sila pastikan pengesahan kata laluan sepadan.');
            $this->redirect('/reset_password.php');
            return;
        }

        $user = $this->passwordResetService->resetPassword($email, $temporaryPassword, $newPassword);
        if (!$user) {
            $this->session->flash('error', 'Kata laluan sementara tidak sah atau telah tamat tempoh.');
            $this->redirect('/reset_password.php');
            return;
        }

        $authenticatedUser = $this->authService->attempt($email, $newPassword);
        if (!$authenticatedUser) {
            $this->session->flash('success', 'Kata laluan baharu telah disimpan. Sila log masuk.');
            $this->redirect('/');
            return;
        }

        $this->session->flash('success', 'Kata laluan anda telah dikemas kini.');
        $this->redirect($this->authService->redirectPath($authenticatedUser));
    }

    private function resetScripts(): string
    {
        return <<<SCRIPT
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script>
    document.addEventListener('DOMContentLoaded', function () {
        if (window.resetFlash && window.Swal) {
            if (window.resetFlash.error) {
                Swal.fire({
                    icon: 'error',
                    title: 'Ralat',
                    text: window.resetFlash.error
                });
            } else if (window.resetFlash.success) {
                Swal.fire({
                    icon: 'success',
                    title: 'Berjaya',
                    text: window.resetFlash.success
                });
            }
        }
    });
</script>
SCRIPT;
    }
}
