<?php

declare(strict_types=1);

namespace App\Core;

use mysqli;
use mysqli_result;
use RuntimeException;

class Database
{
    private static ?Database $instance = null;

    private mysqli $connection;

    private function __construct()
    {
        $config = config('database');
        if (!$config) {
            throw new RuntimeException('Database configuration missing.');
        }

        $host = $config['host'] ?? 'localhost';
        $port = (int) ($config['port'] ?? 3306);
        $username = $config['username'] ?? 'root';
        $password = $config['password'] ?? '';
        $database = $config['database'] ?? '';

        $mysqli = mysqli_init();
        $mysqli->options(MYSQLI_OPT_INT_AND_FLOAT_NATIVE, 1);

        if (!$mysqli->real_connect($host, $username, $password, $database, $port)) {
            throw new RuntimeException('Database connection failed: ' . $mysqli->connect_error);
        }

        $mysqli->set_charset($config['charset'] ?? 'utf8mb4');
        $this->connection = $mysqli;
    }

    public static function connection(): mysqli
    {
        if (!self::$instance) {
            self::$instance = new self();
        }

        return self::$instance->connection;
    }

    public static function query(string $sql, string $types = '', array $params = []): mysqli_result|bool
    {
        $mysqli = self::connection();

        if ($types === '' || !$params) {
            $result = $mysqli->query($sql);
            if ($result === false) {
                throw new RuntimeException('Database query error: ' . $mysqli->error);
            }
            return $result;
        }

        $stmt = $mysqli->prepare($sql);
        if (!$stmt) {
            throw new RuntimeException('Failed to prepare statement: ' . $mysqli->error);
        }

        $stmt->bind_param($types, ...$params);

        if (!$stmt->execute()) {
            $error = $stmt->error;
            $stmt->close();
            throw new RuntimeException('Failed to execute statement: ' . $error);
        }

        $result = $stmt->get_result();
        if ($result instanceof mysqli_result) {
            return $result;
        }

        $stmt->close();
        return true;
    }
}
