<?php

declare(strict_types=1);

namespace App\Services;

use App\Core\Session;
use App\Support\UserRepository;

class AuthService
{
    private Session $session;
    private UserRepository $users;

    public function __construct(Session $session)
    {
        $this->session = $session;
        $this->users = new UserRepository();
    }

    public function attempt(string $email, string $password, bool $persist = true): ?array
    {
        $email = strtolower(trim($email));
        if ($email === '') {
            return null;
        }

        $user = $this->users->findByEmail($email);
        if (!$user) {
            return null;
        }

        if (!password_verify($password, $user['password'] ?? '')) {
            return null;
        }

        unset($user['password']);

        if ($persist) {
            $this->loginUser($user);
        }

        return $user;
    }

    public function loginUser(array $user): void
    {
        $payload = $user;
        unset($payload['password']);

        $this->session->regenerate();
        $this->session->put('user', $payload);
    }

    public function logout(): void
    {
        $this->session->destroy();
    }

    public function redirectPath(?array $user): string
    {
        if (!$user) {
            return '/';
        }

        return match ($user['role'] ?? '') {
            'admin', 'pentadbir', 'pentadbir_agong' => '/dashboard',
            'applicant' => '/dashboard/applicant',
            default => '/dashboard',
        };
    }
}
