<?php

declare(strict_types=1);

use App\Core\Session;

function base_path(string $path = ''): string
{
    $base = __DIR__ . '/..';
    if ($path === '') {
        return $base;
    }
 
    return $base . '/' . ltrim($path, '/');
}

function view_path(string $path = ''): string
{
    $base = base_path('resources/views');
    if ($path === '') {
        return $base;
    }

    return $base . '/' . ltrim($path, '/');
}

function route(string $path): string
{
    $baseUrl = config('app.base_url');
    if ($baseUrl === '' || $baseUrl === null) {
        $scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
        $directory = rtrim(str_replace('\\', '/', dirname($scriptName)), '/');
        if ($directory === '/' || $directory === '\\') {
            $directory = '';
        }
        $baseUrl = $directory;
    }

    return rtrim($baseUrl, '/') . $path;
}

function asset(string $path): string
{
    $assetBase = config('app.asset_base_url');
    if ($assetBase !== null && $assetBase !== '') {
        return rtrim($assetBase, '/') . '/' . ltrim($path, '/');
    }

    return '/' . ltrim($path, '/');
}

function config(string $key, mixed $default = null): mixed
{
    static $config;
    if ($config === null) {
        $config = [
            'app' => require base_path('config/app.php'),
            'database' => require base_path('config/database.php'),
        ];
    }

    $parts = explode('.', $key);
    $group = $parts[0] ?? null;
    $item = $parts[1] ?? null;

    if ($group === null) {
        return $default;
    }

    if (!isset($config[$group])) {
        $path = base_path('config/' . $group . '.php');
        if (is_file($path)) {
            $config[$group] = require $path;
        } else {
            return $default;
        }
    }

    if ($item === null) {
        return $config[$group];
    }

    return $config[$group][$item] ?? $default;
}

function session(): Session
{
    static $session;
    if ($session === null) {
        $session = new Session();
    }
    return $session;
}

function csrf_token(): string
{
    $session = session();
    $token = $session->get('_csrf_token');
    if (!is_string($token) || $token === '') {
        $token = bin2hex(random_bytes(32));
        $session->put('_csrf_token', $token);
    }

    return $token;
}

function csrf_field(): string
{
    $token = htmlspecialchars(csrf_token(), ENT_QUOTES, 'UTF-8');
    return '<input type="hidden" name="_token" value="' . $token . '">';
}

function validate_csrf_token(?string $token): bool
{
    if (!is_string($token) || $token === '') {
        return false;
    }

    $sessionToken = session()->get('_csrf_token');
    return is_string($sessionToken) && $sessionToken !== '' && hash_equals($sessionToken, $token);
}

if (!function_exists('str_starts_with')) {
    function str_starts_with(string $haystack, string $needle): bool
    {
        return substr($haystack, 0, strlen($needle)) === $needle;
    }
}
