<?php

//declare(strict_types=1);

require_once dirname(__DIR__, 2) . '/urus/vendor/phpmailer/phpmailer/src/Exception.php';
require_once dirname(__DIR__, 2) . '/urus/vendor/phpmailer/phpmailer/src/PHPMailer.php';
require_once dirname(__DIR__, 2) . '/urus/vendor/phpmailer/phpmailer/src/SMTP.php';

use PHPMailer\PHPMailer\Exception as PHPMailerException;
use PHPMailer\PHPMailer\PHPMailer;

function pemohon_config(string $key, mixed $default = null): mixed
{
    static $config;
    if ($config === null) {
        $config = require __DIR__ . '/../config.php';
    }

    $segments = explode('.', $key);
    $value = $config;

    foreach ($segments as $segment) {
        if (!is_array($value) || !array_key_exists($segment, $value)) {
            return $default;
        }
        $value = $value[$segment];
    }

    return $value;
}

function pemohon_asset(string $path): string
{
    $base = pemohon_config('asset_base_url', '/');
    return rtrim($base, '/') . '/' . ltrim($path, '/');
}

function pemohon_redirect(string $path): void
{
    header('Location: ' . $path);
    exit;
}

function pemohon_start_session(): void
{
    if (session_status() !== PHP_SESSION_ACTIVE) {
        session_start();
    }
}

function pemohon_csrf_token(): string
{
    pemohon_start_session();

    $token = $_SESSION['_csrf_token'] ?? null;
    if (!is_string($token) || $token === '') {
        $token = bin2hex(random_bytes(32));
        $_SESSION['_csrf_token'] = $token;
    }

    return $token;
}

function pemohon_csrf_field(): string
{
    $token = htmlspecialchars(pemohon_csrf_token(), ENT_QUOTES, 'UTF-8');
    return '<input type="hidden" name="_token" value="' . $token . '">';
}

function pemohon_validate_csrf_token(?string $token): bool
{
    if (!is_string($token) || $token === '') {
        return false;
    }

    pemohon_start_session();
    $sessionToken = $_SESSION['_csrf_token'] ?? null;

    return is_string($sessionToken) && $sessionToken !== '' && hash_equals($sessionToken, $token);
}

function pemohon_resolve_avatar_url(?array $user, ?string $fallback = null): string
{
    $default = $fallback ?? pemohon_asset('assets/images/users/1.jpg');

    if (!is_array($user)) {
        return $default;
    }

    $photoPath = $user['photo_path'] ?? '';
    if (!is_string($photoPath) || $photoPath === '') {
        return $default;
    }

    $normalizedPath = ltrim(str_replace('\\', '/', $photoPath), '/');
    $projectRoot = dirname(__DIR__, 2);
    $absolutePath = $projectRoot . '/' . $normalizedPath;

    if (!is_file($absolutePath)) {
        return $default;
    }

    return pemohon_asset($normalizedPath);
}

function pemohon_generate_uuid(): string
{
    $data = random_bytes(16);
    $data[6] = chr((ord($data[6]) & 0x0f) | 0x40);
    $data[8] = chr((ord($data[8]) & 0x3f) | 0x80);

    return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
}

function pemohon_generate_password(int $length = 12): string
{
    $alphabet = 'ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz23456789@#%&!?';
    $maxIndex = strlen($alphabet) - 1;
    $password = '';

    for ($i = 0; $i < $length; $i++) {
        $password .= $alphabet[random_int(0, $maxIndex)];
    }

    return $password;
}

function pemohon_verify_recaptcha(?string $token, string $action = ''): bool
{
    $secretKey = (string) pemohon_config('recaptcha.secret_key', '');
    if ($secretKey === '' || !is_string($token) || $token === '') {
        // If not configured, treat as passed to avoid blocking legitimate users in development.
        return $secretKey === '' ? true : false;
    }

    $payload = http_build_query([
        'secret' => $secretKey,
        'response' => $token,
    ]);

    $context = stream_context_create([
        'http' => [
            'method' => 'POST',
            'header' => "Content-Type: application/x-www-form-urlencoded\r\n",
            'content' => $payload,
            'timeout' => 10,
        ],
    ]);

    $response = @file_get_contents('https://www.google.com/recaptcha/api/siteverify', false, $context);
    if ($response === false) {
        return false;
    }

    $data = json_decode($response, true);
    if (!is_array($data) || empty($data['success'])) {
        return false;
    }

    if ($action !== '' && isset($data['action']) && $data['action'] !== $action) {
        return false;
    }

    if (isset($data['score']) && is_numeric($data['score']) && (float) $data['score'] < 0.5) {
        return false;
    }

    return true;
}

function pemohon_send_mail(string $to, string $subject, string $htmlBody, ?string $textBody = null): bool
{
    $fromAddress = (string) pemohon_config('mail.from_address', 'noreply@example.com');
    $fromName = (string) pemohon_config('mail.from_name', 'Portal Pemohon eTauliah');
    $host = (string) pemohon_config('mail.host', 'localhost');
    $port = (int) pemohon_config('mail.port', 25);
    $username = (string) pemohon_config('mail.username', '');
    $password = (string) pemohon_config('mail.password', '');
    $encryption = strtolower((string) pemohon_config('mail.encryption', ''));

    $verifyPeer = (bool) pemohon_config('mail.verify_peer', true);
    $allowSelfSigned = (bool) pemohon_config('mail.allow_self_signed', false);

    $mail = new PHPMailer(true);

    try {
        $mail->isSMTP();
        $mail->Host = $host;
        $mail->Port = $port;
        $mail->SMTPAuth = $username !== '' && $password !== '';
        if ($mail->SMTPAuth) {
            $mail->Username = $username;
            $mail->Password = $password;
        }

        if ($encryption === 'tls') {
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
        } elseif ($encryption === 'ssl') {
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
        }

        $mail->SMTPAutoTLS = true;
        if (!$verifyPeer || $allowSelfSigned) {
            $mail->SMTPOptions = [
                'ssl' => [
                    'verify_peer' => $verifyPeer,
                    'verify_peer_name' => $verifyPeer,
                    'allow_self_signed' => $allowSelfSigned,
                ],
            ];
        }
        $mail->CharSet = 'UTF-8';
        $mail->setFrom($fromAddress, $fromName);
        $mail->addAddress($to);
        $mail->Subject = $subject;
        $mail->isHTML(true);
        $mail->Body = $htmlBody;
        $mail->AltBody = $textBody !== null && $textBody !== '' ? $textBody : trim(strip_tags($htmlBody));

        $mail->send();
        return true;
    } catch (PHPMailerException $exception) {
        error_log('pemohon_send_mail error: ' . $exception->getMessage());
        return false;
    }
}
