<?php

declare(strict_types=1);

require_once __DIR__ . '/includes/helpers.php';
require_once __DIR__ . '/includes/auth.php';

pemohon_start_session();

header('Content-Type: application/json; charset=utf-8');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['message' => 'Kaedah tidak dibenarkan.']);
    exit;
}

$token = $_POST['_token'] ?? '';
if (!pemohon_validate_csrf_token(is_string($token) ? $token : null)) {
    http_response_code(419);
    echo json_encode(['message' => 'Sesi tidak sah. Sila muat semula halaman.']);
    exit;
}

$name = trim((string) ($_POST['name'] ?? ''));
$email = strtolower(trim((string) ($_POST['email'] ?? '')));
$emailConfirmation = strtolower(trim((string) ($_POST['email_confirmation'] ?? '')));
$phone = trim((string) ($_POST['phone'] ?? ''));
$recaptchaToken = $_POST['recaptcha_token'] ?? '';

$errors = [];

if ($name === '') {
    $errors['name'] = 'Nama penuh diperlukan.';
}

if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
    $errors['email'] = 'Emel tidak sah.';
}

if ($email !== $emailConfirmation) {
    $errors['email_confirmation'] = 'Emel pengesahan tidak sepadan.';
}

if ($phone === '') {
    $errors['phone'] = 'No. telefon bimbit diperlukan.';
} elseif (!preg_match('/^\+?[0-9\s\-]{6,20}$/', $phone)) {
    $errors['phone'] = 'Sila masukkan nombor telefon bimbit yang sah.';
}

$recaptchaValid = pemohon_verify_recaptcha(is_string($recaptchaToken) ? $recaptchaToken : '', 'signup');
if (!$recaptchaValid) {
    $errors['recaptcha'] = 'Pengesahan reCAPTCHA gagal. Sila cuba lagi.';
}

if ($errors) {
    http_response_code(422);
    echo json_encode([
        'message' => 'Sila betulkan maklumat yang diberikan.',
        'errors' => $errors,
    ]);
    exit;
}

$db = pemohon_db();

$existing = null;
$checkStmt = $db->prepare('SELECT id FROM users WHERE email = ? LIMIT 1');
if ($checkStmt) {
    $checkStmt->bind_param('s', $email);
    $checkStmt->execute();
    $checkStmt->bind_result($existingId);
    if ($checkStmt->fetch()) {
        $existing = $existingId;
    }
    $checkStmt->close();
}

if ($existing) {
    http_response_code(422);
    echo json_encode([
        'message' => 'Emel ini telah didaftarkan.',
        'errors' => ['email' => 'Emel ini telah didaftarkan.'],
    ]);
    exit;
}

$roleId = null;
$roleStmt = $db->prepare('SELECT id FROM roles WHERE name = ? LIMIT 1');
if ($roleStmt) {
    $roleName = 'applicant';
    $roleStmt->bind_param('s', $roleName);
    $roleStmt->execute();
    $roleStmt->bind_result($roleIdValue);
    if ($roleStmt->fetch()) {
        $roleId = (int) $roleIdValue;
    }
    $roleStmt->close();
}

if (!$roleId) {
    http_response_code(500);
    echo json_encode(['message' => 'Konfigurasi peranan pemohon tidak ditemui.']);
    exit;
}

$temporaryPassword = pemohon_generate_password(12);
$hashedPassword = password_hash($temporaryPassword, PASSWORD_DEFAULT);
$uuid = pemohon_generate_uuid();
$status = 'active';

$db->begin_transaction();

try {
    $userStmt = $db->prepare(
        'INSERT INTO users (uuid, name, email, password, phone, status, force_password_change, created_at, updated_at)
         VALUES (?, ?, ?, ?, NULLIF(?, \'\'), ?, 1, NOW(), NOW())'
    );
    if (!$userStmt) {
        throw new RuntimeException('Gagal menyediakan pendaftaran pengguna: ' . $db->error);
    }

    $userStmt->bind_param('ssssss', $uuid, $name, $email, $hashedPassword, $phone, $status);
    $userStmt->execute();
    $userId = (int) $userStmt->insert_id;
    $userStmt->close();

    $roleUserStmt = $db->prepare(
        'INSERT INTO role_user (user_id, role_id, is_primary, assigned_at) VALUES (?, ?, 1, NOW())'
    );
    if (!$roleUserStmt) {
        throw new RuntimeException('Gagal menetapkan peranan pemohon: ' . $db->error);
    }
    $roleUserStmt->bind_param('ii', $userId, $roleId);
    $roleUserStmt->execute();
    $roleUserStmt->close();

    $applicantStmt = $db->prepare('INSERT INTO applicants (user_id, created_at, updated_at) VALUES (?, NOW(), NOW())');
    if (!$applicantStmt) {
        throw new RuntimeException('Gagal menyediakan profil pemohon: ' . $db->error);
    }
    $applicantStmt->bind_param('i', $userId);
    $applicantStmt->execute();
    $applicantStmt->close();

    $db->commit();
} catch (Throwable $exception) {
    $db->rollback();
    http_response_code(500);
    echo json_encode(['message' => 'Pendaftaran tidak dapat diteruskan. Sila cuba lagi.']);
    exit;
}

$loginUrl = pemohon_asset('pemohon/');
$subject = 'Pendaftaran eTauliah - Kata Laluan Sementara';
$escapedName = htmlspecialchars($name, ENT_QUOTES, 'UTF-8');
$escapedEmail = htmlspecialchars($email, ENT_QUOTES, 'UTF-8');
$escapedPassword = htmlspecialchars($temporaryPassword, ENT_QUOTES, 'UTF-8');
$escapedLoginUrl = htmlspecialchars($loginUrl, ENT_QUOTES, 'UTF-8');

$htmlBody = <<<HTML
<p>Hai {$escapedName},</p>
<p>Terima kasih kerana mendaftar sebagai pemohon eTauliah.</p>
<p><strong>Maklumat Log Masuk:</strong><br>
Emel: {$escapedEmail}<br>
Kata Laluan Sementara: <strong>{$escapedPassword}</strong></p>
<p>Sila log masuk di <a href="{$escapedLoginUrl}" target="_blank" rel="noopener">Portal Pemohon eTauliah</a> dan tukar kata laluan kepada yang baharu selepas berjaya log masuk.</p>
<p>Kata laluan sementara ini adalah rahsia. Jangan kongsikan dengan sesiapa.</p>
<p>Terima kasih,<br>Pasukan eTauliah</p>
HTML;

$textBody = <<<TEXT
Hai {$name},

Terima kasih kerana mendaftar sebagai pemohon eTauliah.

Maklumat Log Masuk:
Emel: {$email}
Kata Laluan Sementara: {$temporaryPassword}

Log masuk di: {$loginUrl}
Selepas log masuk, sila tukar kata laluan anda.

Terima kasih,
Pasukan eTauliah
TEXT;

$mailSent = pemohon_send_mail($email, $subject, $htmlBody, $textBody);

if (!$mailSent) {
    http_response_code(500);
    echo json_encode([
        'message' => 'Pendaftaran berjaya tetapi emel tidak dapat dihantar. Sila cuba semula atau hubungi pentadbir.',
    ]);
    exit;
}

echo json_encode([
    'message' => 'Pendaftaran berjaya. Sila semak emel untuk kata laluan sementara.',
]);
exit;
