<?php

declare(strict_types=1);

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Request;
use App\Core\Response;
use App\Core\Session;
use App\Services\AuthService;

class AuthController extends Controller
{
    private const ADMIN_ROLES = ['admin', 'pentadbir', 'pentadbir_agong'];
    private const APPLICANT_ROLES = ['applicant'];

    private AuthService $authService;

    public function __construct(Request $request, Response $response, Session $session)
    {
        parent::__construct($request, $response, $session);
        $this->authService = new AuthService($session);
    }

    public function showAdminLogin(): void
    {
        $this->renderLogin('admin');
    }

    public function showApplicantLogin(): void
    {
        $this->renderLogin('applicant');
    }

    private function renderLogin(string $context): void
    {
        $siteKey = config('app.recaptcha_site_key');
        $title = $context === 'applicant'
            ? 'Portal Pemohon eTauliah'
            : 'Portal Pentadbir eTauliah';

        $this->view('auth.login', [
            'title' => $title,
            'context' => $context,
            'heading' => $title,
            'loginAction' => route('/login'),
            'error' => $this->session->getFlash('error'),
            'success' => $this->session->getFlash('success'),
            'recaptchaSiteKey' => $siteKey,
            'pageScripts' => $this->loginScripts($siteKey),
        ], 'guest');
    }

    public function adminLogin(): void
    {
        $this->handleLogin(self::ADMIN_ROLES, '/');
    }

    public function applicantLogin(): void
    {
        $this->handleLogin(self::APPLICANT_ROLES, '/');
    }

    public function logout(): void
    {
        $this->authService->logout();
        $this->session->flash('success', 'Anda telah log keluar.');
        $this->redirect('/');
    }

    private function loginScripts(?string $siteKey): string
    {
        $sweetAlert = '<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>';
        $zxcvbn = '<script src="https://cdnjs.cloudflare.com/ajax/libs/zxcvbn/4.4.2/zxcvbn.js"></script>';

        $coreScript = <<<SCRIPT
<script>
    document.addEventListener('DOMContentLoaded', function () {
        var form = document.getElementById('loginform');
        var passwordInput = document.getElementById('password');
        var toggleButton = document.getElementById('togglePassword');
        var strengthText = document.getElementById('passwordStrength');
        var submitButton = form ? form.querySelector('button[type="submit"]') : null;

        var showAlert = function (options) {
            if (window.Swal) {
                return Swal.fire(options);
            }
            var text = options && (options.text || options.title) ? (options.text || options.title) : 'Makluman';
            alert(text);
            return Promise.resolve();
        };

        if (toggleButton && passwordInput) {
            var toggleIcon = toggleButton.querySelector('i');
            var toggleLabel = toggleButton.querySelector('.visually-hidden');

            toggleButton.addEventListener('click', function () {
                var isCurrentlyPassword = passwordInput.type === 'password';
                passwordInput.type = isCurrentlyPassword ? 'text' : 'password';
                if (toggleIcon) {
                    toggleIcon.classList.toggle('fa-eye', !isCurrentlyPassword);
                    toggleIcon.classList.toggle('fa-eye-slash', isCurrentlyPassword);
                }
                var ariaText = isCurrentlyPassword ? 'Sembunyikan kata laluan' : 'Tunjukkan kata laluan';
                toggleButton.setAttribute('aria-label', ariaText);
                if (toggleLabel) {
                    toggleLabel.textContent = ariaText;
                }
            });
        }

        var updateStrength = function () {
            if (!strengthText || !passwordInput) {
                return;
            }

            var value = passwordInput.value || '';
            if (value.length === 0) {
                strengthText.textContent = '';
                strengthText.className = 'form-text text-muted mt-2';
                return;
            }

            if (typeof zxcvbn !== 'function') {
                strengthText.textContent = '';
                strengthText.className = 'form-text text-muted mt-2';
                return;
            }

            var result = zxcvbn(value);
            var labels = ['Sangat Lemah', 'Lemah', 'Sederhana', 'Baik', 'Sangat Baik'];
            var classes = ['text-danger', 'text-danger', 'text-warning', 'text-info', 'text-success'];
            var score = Math.max(0, Math.min(result.score, labels.length - 1));

            strengthText.textContent = 'Kekuatan Kata Laluan: ' + labels[score];
            strengthText.className = 'form-text mt-2 ' + classes[score];
        };

        if (passwordInput) {
            passwordInput.addEventListener('input', updateStrength);
            updateStrength();
        }

        if (window.loginFlash) {
            if (window.loginFlash.error) {
                showAlert({
                    icon: 'error',
                    title: 'Ralat',
                    text: window.loginFlash.error
                });
            } else if (window.loginFlash.success) {
                showAlert({
                    icon: 'success',
                    title: 'Berjaya',
                    text: window.loginFlash.success
                });
            }
        }

        var disableSubmit = function () {
            if (!submitButton) {
                return;
            }
            if (!submitButton.dataset.originalText) {
                submitButton.dataset.originalText = submitButton.textContent;
            }
            submitButton.disabled = true;
            submitButton.textContent = 'Menghantar...';
        };

        var enableSubmit = function () {
            if (!submitButton) {
                return;
            }
            submitButton.disabled = false;
            submitButton.textContent = submitButton.dataset.originalText || submitButton.textContent;
        };

        var submitLoginRequest = function () {
            if (!form) {
                return Promise.resolve();
            }

            disableSubmit();

            var formData = new FormData(form);
            var payload = new URLSearchParams();
            formData.forEach(function (value, key) {
                payload.append(key, value.toString());
            });

            return fetch(form.action, {
                method: 'POST',
                credentials: 'same-origin',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded;charset=UTF-8',
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json'
                },
                body: payload.toString()
            }).then(function (response) {
                return response.json().catch(function () {
                    return {};
                }).then(function (data) {
                    return {
                        ok: response.ok,
                        status: response.status,
                        body: data
                    };
                });
            }).then(function (result) {
                enableSubmit();

                var message = result.body && result.body.message ? result.body.message : 'Permintaan tidak dapat diproses.';
                if (result.ok && result.body && result.body.success) {
                    var redirectUrl = result.body.redirect || form.getAttribute('data-success-redirect') || '/';
                    window.location.assign(redirectUrl);
                    return;
                }

                if (result.body && result.body.shouldReload) {
                    return showAlert({
                        icon: 'warning',
                        title: 'Sesi Tamat',
                        text: message
                    }).then(function () {
                        window.location.reload();
                    });
                }

                return showAlert({
                    icon: 'error',
                    title: 'Ralat',
                    text: message
                });
            }).catch(function () {
                enableSubmit();
                return showAlert({
                    icon: 'error',
                    title: 'Ralat',
                    text: 'Permintaan tidak dapat diproses sekarang. Sila cuba lagi.'
                });
            });
        };

        if (form) {
            form.addEventListener('submit', function (event) {
                event.preventDefault();

                var recaptchaPromise = typeof window.executeLoginRecaptcha === 'function'
                    ? window.executeLoginRecaptcha(form)
                    : Promise.resolve();

                recaptchaPromise.catch(function () {
                    return null;
                }).then(function () {
                    submitLoginRequest();
                });
            });
        }

        var forgotPasswordForm = document.getElementById('forgotPasswordForm');
        var forgotPasswordSubmit = document.getElementById('forgotPasswordSubmit');
        var forgotPasswordModalElement = document.getElementById('forgotPasswordModal');
        var forgotPasswordModal = null;

        if (forgotPasswordModalElement && forgotPasswordModalElement.parentNode !== document.body) {
            document.body.appendChild(forgotPasswordModalElement);
        }

        if (forgotPasswordModalElement && window.bootstrap && window.bootstrap.Modal) {
            forgotPasswordModal = window.bootstrap.Modal.getOrCreateInstance(forgotPasswordModalElement);
        }

        if (forgotPasswordForm) {
            forgotPasswordForm.addEventListener('submit', function (event) {
                event.preventDefault();

                if (forgotPasswordSubmit) {
                    if (!forgotPasswordSubmit.dataset.originalText) {
                        forgotPasswordSubmit.dataset.originalText = forgotPasswordSubmit.textContent;
                    }
                    forgotPasswordSubmit.disabled = true;
                    forgotPasswordSubmit.textContent = 'Menghantar...';
                }

                var formData = new FormData(forgotPasswordForm);
                var payload = new URLSearchParams();
                formData.forEach(function (value, key) {
                    payload.append(key, value.toString());
                });

                fetch(forgotPasswordForm.action, {
                    method: 'POST',
                    credentials: 'same-origin',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded;charset=UTF-8',
                        'X-Requested-With': 'XMLHttpRequest',
                        'Accept': 'application/json'
                    },
                    body: payload.toString()
                }).then(function (response) {
                    return response.json().catch(function () {
                        return {};
                    }).then(function (data) {
                        return {
                            ok: response.ok,
                            status: response.status,
                            body: data
                        };
                    });
                }).then(function (result) {
                    if (forgotPasswordSubmit) {
                        forgotPasswordSubmit.disabled = false;
                        forgotPasswordSubmit.textContent = forgotPasswordSubmit.dataset.originalText || 'Hantar';
                    }

                    var message = result.body && result.body.message ? result.body.message : 'Permintaan tamat tanpa mesej.';

                    if (result.ok) {
                        forgotPasswordForm.reset();
                        if (forgotPasswordModal) {
                            forgotPasswordModal.hide();
                        }
                        return showAlert({
                            icon: 'success',
                            title: 'Emel Dihantar',
                            text: message
                        });
                    }

                    return showAlert({
                        icon: result.status === 422 ? 'warning' : 'error',
                        title: 'Tidak Berjaya',
                        text: message
                    });
                }).catch(function () {
                    if (forgotPasswordSubmit) {
                        forgotPasswordSubmit.disabled = false;
                        forgotPasswordSubmit.textContent = forgotPasswordSubmit.dataset.originalText || 'Hantar';
                    }
                    return showAlert({
                        icon: 'error',
                        title: 'Tidak Berjaya',
                        text: 'Permintaan tidak dapat diproses sekarang. Sila cuba lagi.'
                    });
                });
            });
        }
    });
</script>
SCRIPT;

        $recaptchaScript = '';
        if ($siteKey && $siteKey !== 'YOUR_RECAPTCHA_V3_SITE_KEY') {
            $escapedSiteKey = htmlspecialchars($siteKey, ENT_QUOTES, 'UTF-8');
            $recaptchaScript = <<<SCRIPT
<script src="https://www.google.com/recaptcha/api.js?render={$escapedSiteKey}"></script>
<script>
    document.addEventListener('DOMContentLoaded', function () {
        var form = document.getElementById('loginform');
        if (!form) {
            return;
        }

        window.executeLoginRecaptcha = function (activeForm) {
            var targetForm = activeForm || form;

            return new Promise(function (resolve, reject) {
                if (!window.grecaptcha) {
                    resolve();
                    return;
                }

                window.grecaptcha.ready(function () {
                    window.grecaptcha.execute('{$escapedSiteKey}', {action: 'login'}).then(function (token) {
                        var input = targetForm.querySelector('input[name="recaptcha_token"]');
                        if (!input) {
                            input = document.createElement('input');
                            input.type = 'hidden';
                            input.name = 'recaptcha_token';
                            targetForm.appendChild(input);
                        }
                        input.value = token;
                        resolve();
                    }).catch(function (error) {
                        reject(error);
                    });
                });
            });
        };
    });
</script>
SCRIPT;
        }

        return implode("\n", array_filter([$sweetAlert, $zxcvbn, $coreScript, $recaptchaScript]));
    }

    private function handleLogin(array $allowedRoles, string $loginPath): void
    {
        $csrfToken = (string) $this->request->input('_token', '');
        if (!validate_csrf_token($csrfToken)) {
            $this->loginFailure('Sesi tidak sah. Sila cuba lagi.', $loginPath, 419, true);
            return;
        }

        $email = trim((string) $this->request->input('email', ''));
        $password = (string) $this->request->input('password', '');

        if ($email === '' || $password === '') {
            $this->loginFailure('Emel dan kata laluan diperlukan.', $loginPath, 422);
            return;
        }

        $user = $this->authService->attempt($email, $password, false);

        if (!$user) {
            $this->loginFailure('Maklumat log masuk tidak sah.', $loginPath, 401);
            return;
        }

        $role = $user['role'] ?? null;
        if (!in_array($role, $allowedRoles, true)) {
            $this->loginFailure('Akses akaun ini tidak dibenarkan untuk portal tersebut.', $loginPath, 403);
            return;
        }

        if (($user['status'] ?? 'pending') !== 'active') {
            $this->loginFailure('Akaun anda belum aktif. Sila hubungi pentadbir.', $loginPath, 423);
            return;
        }

        $this->authService->loginUser($user);

        $this->loginSuccess($user);
    }

    private function loginFailure(string $message, string $loginPath, int $status, bool $shouldReload = false): void
    {
        if ($this->request->isAjax()) {
            $payload = [
                'success' => false,
                'message' => $message,
            ];

            if ($shouldReload) {
                $payload['shouldReload'] = true;
            }

            $this->response->json($payload, $status);
        }

        $this->session->flash('error', $message);
        $this->redirect($loginPath);
    }

    private function loginSuccess(array $user): void
    {
        $redirectPath = $this->authService->redirectPath($user);
        $message = 'Selamat datang kembali, ' . ($user['name'] ?? 'Pengguna') . '.';

        if ($this->request->isAjax()) {
            $this->response->json([
                'success' => true,
                'message' => $message,
                'redirect' => route($redirectPath),
            ]);
        }

        $this->session->flash('success', $message);
        $this->redirect($redirectPath);
    }
}
