<?php

declare(strict_types=1);

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Request;
use App\Core\Response;
use App\Core\Session;

class ModulePageController extends Controller
{
    /** @var array<string, mixed> */
    private array $modules;

    public function __construct(Request $request, Response $response, Session $session)
    {
        parent::__construct($request, $response, $session);
        $this->modules = config('modules', []);
    }

    public function applicantAccountRegistration(): void
    {
        $this->renderModule('pemohon.account_registration');
    }

    public function applicantApplications(): void
    {
        $this->renderModule('pemohon.applications');
    }

    public function applicantDocuments(): void
    {
        $this->renderModule('pemohon.documents');
    }

    public function applicantStatusNotifications(): void
    {
        $this->renderModule('pemohon.status_notifications');
    }

    public function applicantPayments(): void
    {
        $this->renderModule('pemohon.payments');
    }

    public function applicantDigitalCard(): void
    {
        $this->renderModule('pemohon.digital_card');
    }

    public function secretariatApplicationsReview(): void
    {
        $this->renderModule('urusetia.applications_review');
    }

    public function secretariatApprovals(): void
    {
        $this->renderModule('urusetia.approvals');
    }

    public function secretariatInterviewSchedule(): void
    {
        $this->renderModule('urusetia.interview_schedule');
    }

    public function secretariatReports(): void
    {
        $this->renderModule('urusetia.reports');
    }

    public function secretariatActivityLog(): void
    {
        $this->renderModule('urusetia.activity_log');
    }

    public function secretariatCertificateVerification(): void
    {
        $this->renderModule('urusetia.certificate_verification');
    }

    public function panelInterviews(): void
    {
        $this->renderModule('panel.interviews');
    }

    public function panelEvaluations(): void
    {
        $this->renderModule('panel.evaluations');
    }

    public function panelComments(): void
    {
        $this->renderModule('panel.comments');
    }

    public function panelMeetings(): void
    {
        $this->renderModule('panel.meetings');
    }

    public function panelDigitalSignature(): void
    {
        $this->renderModule('panel.digital_signature');
    }

    public function panelDecisionReport(): void
    {
        $this->renderModule('panel.decision_report');
    }

    public function systemUsers(): void
    {
        $this->renderModule('sistem.users');
    }

    public function systemSettings(): void
    {
        $this->renderModule('sistem.settings');
    }

    public function systemNotifications(): void
    {
        $this->renderModule('sistem.notifications');
    }

    public function systemAuditTrail(): void
    {
        $this->renderModule('sistem.audit_trail');
    }

    public function systemBackup(): void
    {
        $this->renderModule('sistem.backup');
    }

    public function systemIntegrations(): void
    {
        $this->renderModule('sistem.integrations');
    }

    public function systemPublicVerification(): void
    {
        $this->renderModule('sistem.public_verification');
    }

    public function systemPaymentReport(): void
    {
        $this->renderModule('sistem.payment_report');
    }

    public function reportsAnalytics(): void
    {
        $this->renderModule('laporan.analytics');
    }

    public function reportsExport(): void
    {
        $this->renderModule('laporan.export');
    }

    public function reportsDataLake(): void
    {
        $this->renderModule('laporan.data_lake');
    }

    private function renderModule(string $key): void
    {
        $module = $this->findModule($key);

        if ($module === null) {
            $this->session->flash('error', 'Modul yang diminta tidak ditemui.');
            $this->redirect('/dashboard');
            return;
        }

        $user = $this->session->get('user') ?? [];
        $role = $user['role'] ?? null;
        $allowedRoles = $module['item']['roles'] ?? [];

        if ($allowedRoles !== [] && ($role === null || !in_array($role, $allowedRoles, true))) {
            if ($this->request->isAjax()) {
                $this->response->json([
                    'message' => 'Akses tidak dibenarkan untuk modul ini.',
                ], 403);
                return;
            }

            $this->session->flash('error', 'Akses tidak dibenarkan untuk modul ini.');
            $this->redirect('/dashboard');
            return;
        }

        $this->view('modules.placeholder', [
            'title' => $module['item']['label'],
            'module' => $module['item'],
            'parentModule' => $module['parent'],
        ]);
    }

    /**
     * @return array<string, mixed>|null
     */
    private function findModule(string $key): ?array
    {
        foreach ($this->modules as $parentKey => $parent) {
            $items = $parent['items'] ?? [];
            foreach ($items as $item) {
                if (($item['key'] ?? null) === $key) {
                    $item['roles'] = $item['roles'] ?? ($parent['roles'] ?? []);
                    return [
                        'parent' => [
                            'key' => $parentKey,
                            'label' => $parent['label'] ?? (string) $parentKey,
                            'icon' => $parent['icon'] ?? 'fas fa-folder',
                            'description' => $parent['description'] ?? '',
                        ],
                        'item' => $item,
                    ];
                }
            }
        }

        return null;
    }
}
