<?php

declare(strict_types=1);

namespace App\Core;

use Closure;

class Router
{
    /** @var array<int, array<string, mixed>> */
    private array $routes = [];

    public function add(string $method, string $path, array $handler, ?string $name = null, array $middleware = []): void
    {
        $this->routes[] = [
            'method' => strtoupper($method),
            'path' => $this->normalizePath($path),
            'handler' => $handler,
            'name' => $name,
            'middleware' => $middleware,
        ];
    }

    public function addRoutes(array $routes): void
    {
        foreach ($routes as $route) {
            $this->add(
                $route['method'] ?? 'GET',
                $route['path'] ?? '/',
                $route['handler'],
                $route['name'] ?? null,
                $route['middleware'] ?? []
            );
        }
    }

    public function dispatch(Request $request, Response $response, Session $session): void
    {
        foreach ($this->routes as $route) {
            if ($route['method'] !== $request->method()) {
                continue;
            }

            if ($route['path'] !== $request->uri()) {
                continue;
            }

            $this->handleMiddleware($route['middleware'], $request, $response, $session);
            $handler = $route['handler'];
            [$class, $method] = $handler;
            $controller = new $class($request, $response, $session);
            $controller->{$method}();
            return;
        }

        http_response_code(404);
        echo '404 Not Found';
    }

    private function normalizePath(string $path): string
    {
        $path = trim($path) ?: '/';
        if ($path === '/') {
            return $path;
        }

        return '/' . trim($path, '/');
    }

    private function handleMiddleware(array $middleware, Request $request, Response $response, Session $session): void
    {
        foreach ($middleware as $item) {
            if ($item === 'guest' && $session->get('user')) {
                $response->redirect('/dashboard');
            }

            if (str_starts_with($item, 'auth')) {
                if (!$session->get('user')) {
                    $session->flash('error', 'Sila log masuk untuk meneruskan.');
                    $response->redirect('/');
                }

                $parts = explode(':', $item);
                if (isset($parts[1])) {
                    $allowed = explode(',', $parts[1]);
                    $user = $session->get('user');
                    if (!in_array($user['role'] ?? '', $allowed, true)) {
                        $response->redirect('/dashboard');
                    }
                }
            }
        }
    }
}
